<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

initSecureSession();
requireLogin();

if (!checkSessionTimeout($pdo)) {
    header("Location: login.php?msg=Session+expired");
    exit();
}

$user = getUser($pdo);
if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Get referral stats
$refStats = getReferralStats($pdo, $user['id']);

// Get referred users
$stmt = $pdo->prepare("SELECT u.name, u.created_at, r.bonus_given, r.referrer_bonus 
    FROM referrals r 
    JOIN users u ON r.referred_id = u.id 
    WHERE r.referrer_id = ? 
    ORDER BY r.created_at DESC");
$stmt->execute([$user['id']]);
$referredUsers = $stmt->fetchAll(PDO::FETCH_ASSOC);

// Site URL for sharing
$siteUrl = (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? "https" : "http") . "://{$_SERVER['HTTP_HOST']}";
$referralLink = $siteUrl . "/register.php?ref=" . $refStats['referral_code'];

$referrerBonus = getSetting($pdo, 'referral_bonus_referrer', '50');
$referredBonus = getSetting($pdo, 'referral_bonus_referred', '25');
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Refer & Earn - Satta King</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
            padding-bottom: 80px;
        }
        .header {
            background: linear-gradient(135deg, #8b0000 0%, #1a1a1a 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        .back-btn { color: #fff; text-decoration: none; font-size: 24px; }
        .header-title { font-size: 18px; font-weight: 600; }
        .container { padding: 20px; max-width: 600px; margin: 0 auto; }
        
        .hero-card {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 16px;
            padding: 30px 20px;
            text-align: center;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        .hero-icon { font-size: 60px; margin-bottom: 15px; }
        .hero-title { font-size: 24px; color: #ffd700; margin-bottom: 10px; }
        .hero-desc { color: #888; font-size: 14px; line-height: 1.6; }
        
        .bonus-box {
            display: flex;
            justify-content: space-around;
            margin: 20px 0;
            padding: 15px;
            background: #0a0a0a;
            border-radius: 12px;
        }
        .bonus-item { text-align: center; }
        .bonus-amount { font-size: 28px; color: #00ff88; font-weight: bold; }
        .bonus-label { font-size: 12px; color: #888; }
        
        .code-section {
            background: #1a1a1a;
            border-radius: 12px;
            padding: 20px;
            margin-bottom: 20px;
        }
        .code-label { font-size: 14px; color: #888; margin-bottom: 10px; }
        .code-box {
            display: flex;
            gap: 10px;
        }
        .code-display {
            flex: 1;
            background: #0f0f0f;
            border: 2px dashed #ffd700;
            border-radius: 8px;
            padding: 15px;
            text-align: center;
            font-size: 24px;
            font-weight: bold;
            color: #ffd700;
            letter-spacing: 3px;
        }
        .copy-btn {
            background: linear-gradient(135deg, #ffd700, #ff8c00);
            border: none;
            border-radius: 8px;
            padding: 0 20px;
            font-size: 18px;
            cursor: pointer;
        }
        
        .link-section {
            margin-top: 15px;
        }
        .link-box {
            background: #0f0f0f;
            border-radius: 8px;
            padding: 12px;
            font-size: 12px;
            color: #888;
            word-break: break-all;
        }
        
        .share-btns {
            display: flex;
            gap: 10px;
            margin-top: 15px;
        }
        .share-btn {
            flex: 1;
            padding: 12px;
            border: none;
            border-radius: 8px;
            font-size: 14px;
            font-weight: bold;
            cursor: pointer;
            text-decoration: none;
            text-align: center;
        }
        .share-btn.whatsapp { background: #25d366; color: #fff; }
        .share-btn.telegram { background: #0088cc; color: #fff; }
        
        .stats-grid {
            display: grid;
            grid-template-columns: repeat(3, 1fr);
            gap: 10px;
            margin-bottom: 20px;
        }
        .stat-card {
            background: #1a1a1a;
            border-radius: 12px;
            padding: 15px;
            text-align: center;
        }
        .stat-value { font-size: 24px; font-weight: bold; color: #ffd700; }
        .stat-label { font-size: 11px; color: #888; margin-top: 5px; }
        
        .referred-list {
            background: #1a1a1a;
            border-radius: 12px;
            padding: 15px;
        }
        .section-title { font-size: 16px; color: #ffd700; margin-bottom: 15px; }
        .referred-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px;
            background: #0f0f0f;
            border-radius: 8px;
            margin-bottom: 8px;
        }
        .referred-name { font-weight: 500; }
        .referred-date { font-size: 12px; color: #888; }
        .referred-status {
            font-size: 12px;
            padding: 4px 8px;
            border-radius: 4px;
        }
        .referred-status.earned { background: #00ff8822; color: #00ff88; }
        .referred-status.pending { background: #ffd70022; color: #ffd700; }
        .empty-state { text-align: center; color: #888; padding: 30px; }
        
        /* Bottom Navigation */
        .bottom-nav {
            position: fixed;
            bottom: 0;
            left: 0;
            right: 0;
            background: #1a1a1a;
            display: flex;
            justify-content: space-around;
            padding: 10px 0;
            border-top: 1px solid #333;
        }
        .nav-item {
            text-decoration: none;
            color: #888;
            text-align: center;
            font-size: 12px;
        }
        .nav-item.active { color: #ffd700; }
        .nav-icon { font-size: 20px; display: block; margin-bottom: 2px; }
    </style>
</head>
<body>
    <div class="header">
        <a href="index.php" class="back-btn">←</a>
        <span class="header-title">🎁 Refer & Earn</span>
        <span></span>
    </div>
    
    <div class="container">
        <!-- Hero Card -->
        <div class="hero-card">
            <div class="hero-icon">🎁</div>
            <div class="hero-title">Invite Friends & Earn!</div>
            <div class="hero-desc">Share your referral code with friends. When they join and make their first deposit, you both earn bonus!</div>
            
            <div class="bonus-box">
                <div class="bonus-item">
                    <div class="bonus-amount">₹<?= $referrerBonus ?></div>
                    <div class="bonus-label">You Earn</div>
                </div>
                <div class="bonus-item">
                    <div class="bonus-amount">₹<?= $referredBonus ?></div>
                    <div class="bonus-label">Friend Gets</div>
                </div>
            </div>
        </div>
        
        <!-- Referral Code -->
        <div class="code-section">
            <div class="code-label">Your Referral Code</div>
            <div class="code-box">
                <div class="code-display" id="refCode"><?= htmlspecialchars($refStats['referral_code']) ?></div>
                <button class="copy-btn" onclick="copyCode()">📋</button>
            </div>
            
            <div class="link-section">
                <div class="code-label">Or share this link</div>
                <div class="link-box" id="refLink"><?= htmlspecialchars($referralLink) ?></div>
            </div>
            
            <div class="share-btns">
                <a href="https://wa.me/?text=<?= urlencode("🎰 Join Satta King and get ₹{$referredBonus} bonus! Use my code: {$refStats['referral_code']}\n\n$referralLink") ?>" class="share-btn whatsapp" target="_blank">
                    📱 WhatsApp
                </a>
                <a href="https://t.me/share/url?url=<?= urlencode($referralLink) ?>&text=<?= urlencode("🎰 Join Satta King and get ₹{$referredBonus} bonus! Use my code: {$refStats['referral_code']}") ?>" class="share-btn telegram" target="_blank">
                    ✈️ Telegram
                </a>
            </div>
        </div>
        
        <!-- Stats -->
        <div class="stats-grid">
            <div class="stat-card">
                <div class="stat-value"><?= $refStats['total_referrals'] ?></div>
                <div class="stat-label">Total Referrals</div>
            </div>
            <div class="stat-card">
                <div class="stat-value">₹<?= number_format($refStats['total_earned'], 0) ?></div>
                <div class="stat-label">Total Earned</div>
            </div>
            <div class="stat-card">
                <div class="stat-value"><?= count($referredUsers) ?></div>
                <div class="stat-label">Friends Joined</div>
            </div>
        </div>
        
        <!-- Referred Users List -->
        <div class="referred-list">
            <div class="section-title">👥 Your Referrals</div>
            <?php if (empty($referredUsers)): ?>
                <div class="empty-state">
                    <div style="font-size:40px;margin-bottom:10px;">👥</div>
                    <p>No referrals yet. Share your code to start earning!</p>
                </div>
            <?php else: ?>
                <?php foreach ($referredUsers as $ref): ?>
                <div class="referred-item">
                    <div>
                        <div class="referred-name"><?= htmlspecialchars($ref['name']) ?></div>
                        <div class="referred-date"><?= date('d M Y', strtotime($ref['created_at'])) ?></div>
                    </div>
                    <?php if ($ref['bonus_given']): ?>
                        <span class="referred-status earned">+₹<?= number_format($ref['referrer_bonus'], 0) ?></span>
                    <?php else: ?>
                        <span class="referred-status pending">Pending</span>
                    <?php endif; ?>
                </div>
                <?php endforeach; ?>
            <?php endif; ?>
        </div>
    </div>
    
    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <a href="index.php" class="nav-item">
            <span class="nav-icon">🏠</span>
            Home
        </a>
        <a href="wallet.php" class="nav-item">
            <span class="nav-icon">💰</span>
            Wallet
        </a>
        <a href="referral.php" class="nav-item active">
            <span class="nav-icon">🎁</span>
            Refer
        </a>
        <a href="notifications.php" class="nav-item">
            <span class="nav-icon">🔔</span>
            Alerts
        </a>
        <a href="profile.php" class="nav-item">
            <span class="nav-icon">👤</span>
            Profile
        </a>
    </nav>
    
    <script>
        function copyCode() {
            const code = document.getElementById('refCode').textContent;
            navigator.clipboard.writeText(code).then(() => {
                alert('Referral code copied!');
            });
        }
    </script>
</body>
</html>
