<?php
/**
 * Security Functions
 * CSRF Protection, Rate Limiting, Session Security, Brute Force Protection
 * Satta King Platform
 */

// =============================================
// CSRF PROTECTION
// =============================================

/**
 * Generate CSRF Token
 */
function generateCSRFToken() {
    if (!isset($_SESSION['csrf_token']) || !isset($_SESSION['csrf_token_time']) || 
        (time() - $_SESSION['csrf_token_time']) > 3600) {
        $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
        $_SESSION['csrf_token_time'] = time();
    }
    return $_SESSION['csrf_token'];
}

/**
 * Verify CSRF Token
 */
function verifyCSRFToken($token) {
    if (!isset($_SESSION['csrf_token']) || empty($token)) {
        return false;
    }
    return hash_equals($_SESSION['csrf_token'], $token);
}

/**
 * CSRF Hidden Input Field
 */
function csrfField() {
    return '<input type="hidden" name="csrf_token" value="' . generateCSRFToken() . '">';
}

/**
 * Check CSRF on POST requests
 */
function checkCSRF() {
    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        if (!isset($_POST['csrf_token']) || !verifyCSRFToken($_POST['csrf_token'])) {
            // Log security event
            logSecurityEvent(null, 'CSRF_FAILURE', 'high', 'CSRF token validation failed');
            die('<div style="background:#ff4444;color:white;padding:20px;text-align:center;font-family:sans-serif;">
                <h2>⚠️ Security Error</h2>
                <p>Invalid request. Please refresh the page and try again.</p>
                <a href="javascript:history.back()" style="color:white;">Go Back</a>
            </div>');
        }
    }
}

// =============================================
// RATE LIMITING
// =============================================

/**
 * Check Rate Limit
 * @param PDO $pdo Database connection
 * @param string $identifier User ID or IP
 * @param string $action Action type (bid, withdraw, login, etc.)
 * @param int $limit Maximum allowed in window
 * @param int $windowSeconds Time window in seconds
 * @return array ['allowed' => bool, 'remaining' => int, 'reset_in' => int]
 */
function checkRateLimit($pdo, $identifier, $action, $limit, $windowSeconds = 3600) {
    try {
        // Clean old records
        $stmt = $pdo->prepare("DELETE FROM rate_limits WHERE window_start < DATE_SUB(NOW(), INTERVAL ? SECOND)");
        $stmt->execute([$windowSeconds]);
        
        // Get current count
        $stmt = $pdo->prepare("SELECT count, window_start FROM rate_limits WHERE identifier = ? AND action_type = ? AND window_start > DATE_SUB(NOW(), INTERVAL ? SECOND)");
        $stmt->execute([$identifier, $action, $windowSeconds]);
        $record = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($record) {
            $count = $record['count'];
            $windowStart = strtotime($record['window_start']);
            $resetIn = $windowSeconds - (time() - $windowStart);
            
            if ($count >= $limit) {
                return ['allowed' => false, 'remaining' => 0, 'reset_in' => $resetIn];
            }
            
            // Increment count
            $stmt = $pdo->prepare("UPDATE rate_limits SET count = count + 1 WHERE identifier = ? AND action_type = ?");
            $stmt->execute([$identifier, $action]);
            
            return ['allowed' => true, 'remaining' => $limit - $count - 1, 'reset_in' => $resetIn];
        } else {
            // Create new record
            $stmt = $pdo->prepare("INSERT INTO rate_limits (identifier, action_type, count, window_start) VALUES (?, ?, 1, NOW()) ON DUPLICATE KEY UPDATE count = 1, window_start = NOW()");
            $stmt->execute([$identifier, $action]);
            
            return ['allowed' => true, 'remaining' => $limit - 1, 'reset_in' => $windowSeconds];
        }
    } catch (Exception $e) {
        return ['allowed' => true, 'remaining' => $limit, 'reset_in' => $windowSeconds];
    }
}

/**
 * Get Rate Limit Status (without incrementing)
 */
function getRateLimitStatus($pdo, $identifier, $action, $limit, $windowSeconds = 3600) {
    try {
        $stmt = $pdo->prepare("SELECT count, window_start FROM rate_limits WHERE identifier = ? AND action_type = ? AND window_start > DATE_SUB(NOW(), INTERVAL ? SECOND)");
        $stmt->execute([$identifier, $action, $windowSeconds]);
        $record = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($record) {
            $count = $record['count'];
            $windowStart = strtotime($record['window_start']);
            $resetIn = $windowSeconds - (time() - $windowStart);
            return ['count' => $count, 'limit' => $limit, 'remaining' => max(0, $limit - $count), 'reset_in' => $resetIn];
        }
        
        return ['count' => 0, 'limit' => $limit, 'remaining' => $limit, 'reset_in' => $windowSeconds];
    } catch (Exception $e) {
        return ['count' => 0, 'limit' => $limit, 'remaining' => $limit, 'reset_in' => $windowSeconds];
    }
}

// =============================================
// BRUTE FORCE PROTECTION
// =============================================

/**
 * Record Login Attempt
 */
function recordLoginAttempt($pdo, $mobile, $success = false) {
    try {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $stmt = $pdo->prepare("INSERT INTO login_attempts (ip_address, mobile, success) VALUES (?, ?, ?)");
        $stmt->execute([$ip, $mobile, $success ? 1 : 0]);
        
        if (!$success) {
            // Update user's failed attempts
            $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = failed_login_attempts + 1 WHERE mobile = ?");
            $stmt->execute([$mobile]);
            
            // Check if should lock account
            $maxAttempts = getSetting($pdo, 'max_login_attempts', 5);
            $stmt = $pdo->prepare("SELECT failed_login_attempts FROM users WHERE mobile = ?");
            $stmt->execute([$mobile]);
            $attempts = $stmt->fetchColumn();
            
            if ($attempts >= $maxAttempts) {
                $lockoutMinutes = getSetting($pdo, 'lockout_duration_minutes', 30);
                $stmt = $pdo->prepare("UPDATE users SET locked_until = DATE_ADD(NOW(), INTERVAL ? MINUTE) WHERE mobile = ?");
                $stmt->execute([$lockoutMinutes, $mobile]);
                
                logSecurityEvent(null, 'ACCOUNT_LOCKED', 'high', "Account locked after $attempts failed attempts: $mobile", $mobile);
            }
        } else {
            // Reset failed attempts on success
            $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = 0, locked_until = NULL, last_login = NOW(), last_login_ip = ? WHERE mobile = ?");
            $stmt->execute([$ip, $mobile]);
        }
    } catch (Exception $e) {
        // Silent fail
    }
}

/**
 * Check if Account is Locked
 */
function isAccountLocked($pdo, $mobile) {
    try {
        $stmt = $pdo->prepare("SELECT locked_until FROM users WHERE mobile = ?");
        $stmt->execute([$mobile]);
        $lockedUntil = $stmt->fetchColumn();
        
        if ($lockedUntil && strtotime($lockedUntil) > time()) {
            $remainingMinutes = ceil((strtotime($lockedUntil) - time()) / 60);
            return ['locked' => true, 'remaining_minutes' => $remainingMinutes];
        }
        
        return ['locked' => false];
    } catch (Exception $e) {
        return ['locked' => false];
    }
}

/**
 * Check IP-based Rate Limit for Login
 */
function checkLoginRateLimit($pdo) {
    try {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        
        // Check attempts in last 15 minutes
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM login_attempts WHERE ip_address = ? AND attempt_time > DATE_SUB(NOW(), INTERVAL 15 MINUTE) AND success = 0");
        $stmt->execute([$ip]);
        $attempts = $stmt->fetchColumn();
        
        if ($attempts >= 10) {
            logSecurityEvent(null, 'IP_RATE_LIMITED', 'medium', "IP blocked after $attempts login attempts");
            return ['allowed' => false, 'message' => 'Too many login attempts. Please try again after 15 minutes.'];
        }
        
        return ['allowed' => true];
    } catch (Exception $e) {
        return ['allowed' => true];
    }
}

// =============================================
// SESSION SECURITY
// =============================================

/**
 * Initialize Secure Session
 */
function initSecureSession() {
    if (session_status() === PHP_SESSION_NONE) {
        // Secure session settings
        ini_set('session.cookie_httponly', 1);
        ini_set('session.use_only_cookies', 1);
        ini_set('session.cookie_samesite', 'Strict');
        
        if (isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on') {
            ini_set('session.cookie_secure', 1);
        }
        
        session_start();
    }
    
    // Regenerate session ID periodically
    if (!isset($_SESSION['session_created'])) {
        $_SESSION['session_created'] = time();
    } elseif (time() - $_SESSION['session_created'] > 1800) { // 30 minutes
        session_regenerate_id(true);
        $_SESSION['session_created'] = time();
    }
}

/**
 * Check Session Timeout
 */
function checkSessionTimeout($pdo, $timeoutMinutes = null) {
    if ($timeoutMinutes === null) {
        $timeoutMinutes = getSetting($pdo, 'session_timeout_minutes', 60);
    }
    
    if (isset($_SESSION['last_activity'])) {
        $inactive = time() - $_SESSION['last_activity'];
        if ($inactive > ($timeoutMinutes * 60)) {
            // Session expired
            session_unset();
            session_destroy();
            return false;
        }
    }
    
    $_SESSION['last_activity'] = time();
    return true;
}

/**
 * Track User Session
 */
function trackUserSession($pdo, $userId) {
    try {
        $sessionId = session_id();
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        $device = detectDevice($userAgent);
        
        // Update or insert session
        $stmt = $pdo->prepare("INSERT INTO user_sessions (user_id, session_id, ip_address, user_agent, device_info) 
            VALUES (?, ?, ?, ?, ?) 
            ON DUPLICATE KEY UPDATE last_activity = NOW(), ip_address = ?, is_active = 1");
        $stmt->execute([$userId, $sessionId, $ip, $userAgent, $device, $ip]);
        
        // Deactivate old sessions (optional: limit to 3 active sessions)
        // $stmt = $pdo->prepare("UPDATE user_sessions SET is_active = 0 WHERE user_id = ? AND session_id != ? AND is_active = 1 ORDER BY last_activity ASC LIMIT 10");
        // $stmt->execute([$userId, $sessionId]);
        
    } catch (Exception $e) {
        // Silent fail
    }
}

/**
 * Detect Device Type
 */
function detectDevice($userAgent) {
    if (preg_match('/mobile|android|iphone|ipad|tablet/i', $userAgent)) {
        if (preg_match('/iphone|ipad/i', $userAgent)) {
            return 'iOS Device';
        }
        return 'Android/Mobile';
    }
    return 'Desktop';
}

/**
 * Get User Active Sessions
 */
function getUserSessions($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("SELECT * FROM user_sessions WHERE user_id = ? AND is_active = 1 ORDER BY last_activity DESC");
        $stmt->execute([$userId]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Logout Other Sessions
 */
function logoutOtherSessions($pdo, $userId, $currentSessionId) {
    try {
        $stmt = $pdo->prepare("UPDATE user_sessions SET is_active = 0 WHERE user_id = ? AND session_id != ?");
        $stmt->execute([$userId, $currentSessionId]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

// =============================================
// SECURITY LOGGING
// =============================================

/**
 * Log Security Event
 */
function logSecurityEvent($userId, $eventType, $severity = 'low', $details = '', $extra = null) {
    global $pdo;
    try {
        $ip = $_SERVER['REMOTE_ADDR'] ?? 'unknown';
        $userAgent = $_SERVER['HTTP_USER_AGENT'] ?? 'unknown';
        
        if ($extra) {
            $details .= " | Extra: $extra";
        }
        
        $stmt = $pdo->prepare("INSERT INTO security_log (user_id, event_type, severity, ip_address, user_agent, details) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $eventType, $severity, $ip, $userAgent, $details]);
    } catch (Exception $e) {
        // Silent fail
    }
}

// =============================================
// TRANSACTION PIN
// =============================================

/**
 * Set Transaction PIN
 */
function setTransactionPin($pdo, $userId, $pin) {
    if (strlen($pin) !== 4 || !ctype_digit($pin)) {
        return ['success' => false, 'message' => 'PIN must be exactly 4 digits'];
    }
    
    try {
        $hashedPin = password_hash($pin, PASSWORD_DEFAULT);
        $stmt = $pdo->prepare("UPDATE users SET transaction_pin = ? WHERE id = ?");
        $stmt->execute([$hashedPin, $userId]);
        return ['success' => true, 'message' => 'Transaction PIN set successfully'];
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error setting PIN'];
    }
}

/**
 * Verify Transaction PIN
 */
function verifyTransactionPin($pdo, $userId, $pin) {
    try {
        $stmt = $pdo->prepare("SELECT transaction_pin FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $hashedPin = $stmt->fetchColumn();
        
        if (!$hashedPin) {
            return ['valid' => false, 'message' => 'Transaction PIN not set', 'not_set' => true];
        }
        
        if (password_verify($pin, $hashedPin)) {
            return ['valid' => true];
        }
        
        return ['valid' => false, 'message' => 'Invalid PIN'];
    } catch (Exception $e) {
        return ['valid' => false, 'message' => 'Error verifying PIN'];
    }
}

/**
 * Check if Transaction PIN is Set
 */
function hasTransactionPin($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("SELECT transaction_pin FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $pin = $stmt->fetchColumn();
        return !empty($pin);
    } catch (Exception $e) {
        return false;
    }
}

// =============================================
// DAILY LIMITS
// =============================================

/**
 * Check Daily Withdraw Limit
 */
function checkDailyWithdrawLimit($pdo, $userId, $amount) {
    try {
        $dailyLimit = floatval(getSetting($pdo, 'daily_withdraw_limit', 50000));
        
        // Get today's total
        $stmt = $pdo->prepare("SELECT daily_withdraw_total, daily_withdraw_date FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $today = date('Y-m-d');
        $currentTotal = 0;
        
        if ($user['daily_withdraw_date'] === $today) {
            $currentTotal = floatval($user['daily_withdraw_total']);
        }
        
        $newTotal = $currentTotal + $amount;
        
        if ($newTotal > $dailyLimit) {
            $remaining = max(0, $dailyLimit - $currentTotal);
            return [
                'allowed' => false, 
                'message' => "Daily limit exceeded! Today's limit: ₹" . number_format($dailyLimit, 0) . ". Remaining: ₹" . number_format($remaining, 0),
                'remaining' => $remaining
            ];
        }
        
        return ['allowed' => true, 'remaining' => $dailyLimit - $newTotal];
    } catch (Exception $e) {
        return ['allowed' => true, 'remaining' => 50000];
    }
}

/**
 * Update Daily Withdraw Total
 */
function updateDailyWithdrawTotal($pdo, $userId, $amount) {
    try {
        $today = date('Y-m-d');
        
        $stmt = $pdo->prepare("SELECT daily_withdraw_date FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $currentDate = $stmt->fetchColumn();
        
        if ($currentDate === $today) {
            $stmt = $pdo->prepare("UPDATE users SET daily_withdraw_total = daily_withdraw_total + ? WHERE id = ?");
        } else {
            $stmt = $pdo->prepare("UPDATE users SET daily_withdraw_total = ?, daily_withdraw_date = ? WHERE id = ?");
            $stmt->execute([$amount, $today, $userId]);
            return;
        }
        $stmt->execute([$amount, $userId]);
    } catch (Exception $e) {
        // Silent fail
    }
}

/**
 * Check Daily Bid Limit
 */
function checkDailyBidLimit($pdo, $userId, $amount) {
    try {
        $dailyLimit = floatval(getSetting($pdo, 'daily_bid_limit', 100000));
        
        $stmt = $pdo->prepare("SELECT daily_bid_total, daily_bid_date FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        $today = date('Y-m-d');
        $currentTotal = 0;
        
        if ($user['daily_bid_date'] === $today) {
            $currentTotal = floatval($user['daily_bid_total']);
        }
        
        $newTotal = $currentTotal + $amount;
        
        if ($newTotal > $dailyLimit) {
            $remaining = max(0, $dailyLimit - $currentTotal);
            return [
                'allowed' => false,
                'message' => "Daily bid limit exceeded! Limit: ₹" . number_format($dailyLimit, 0) . ". Remaining: ₹" . number_format($remaining, 0),
                'remaining' => $remaining
            ];
        }
        
        return ['allowed' => true, 'remaining' => $dailyLimit - $newTotal];
    } catch (Exception $e) {
        return ['allowed' => true, 'remaining' => 100000];
    }
}

/**
 * Update Daily Bid Total
 */
function updateDailyBidTotal($pdo, $userId, $amount) {
    try {
        $today = date('Y-m-d');
        
        $stmt = $pdo->prepare("SELECT daily_bid_date FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $currentDate = $stmt->fetchColumn();
        
        if ($currentDate === $today) {
            $stmt = $pdo->prepare("UPDATE users SET daily_bid_total = daily_bid_total + ? WHERE id = ?");
            $stmt->execute([$amount, $userId]);
        } else {
            $stmt = $pdo->prepare("UPDATE users SET daily_bid_total = ?, daily_bid_date = ? WHERE id = ?");
            $stmt->execute([$amount, $today, $userId]);
        }
    } catch (Exception $e) {
        // Silent fail
    }
}

// =============================================
// BID CANCELLATION
// =============================================

/**
 * Check if Bid Can Be Cancelled
 */
function canCancelBid($pdo, $bidId, $userId) {
    try {
        $cancelMinutes = intval(getSetting($pdo, 'bid_cancel_minutes', 5));
        
        $stmt = $pdo->prepare("
            SELECT b.*, g.close_time, g.is_overnight 
            FROM bids b 
            JOIN games g ON b.game_id = g.id 
            WHERE b.id = ? AND b.user_id = ? AND b.status = 'pending' AND b.is_cancelled = 0
        ");
        $stmt->execute([$bidId, $userId]);
        $bid = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$bid) {
            return ['can_cancel' => false, 'message' => 'Bid not found or already processed'];
        }
        
        // Check time
        $closeTime = $bid['close_time'];
        $now = new DateTime();
        $closeDateTime = new DateTime($bid['playing_date'] . ' ' . $closeTime);
        
        // For overnight games
        if ($bid['is_overnight'] && $closeTime < '12:00:00') {
            $closeDateTime->modify('+1 day');
        }
        
        $diff = $closeDateTime->getTimestamp() - $now->getTimestamp();
        $minutesLeft = $diff / 60;
        
        if ($minutesLeft < $cancelMinutes) {
            return ['can_cancel' => false, 'message' => "Cannot cancel within $cancelMinutes minutes of close time"];
        }
        
        return ['can_cancel' => true, 'bid' => $bid];
    } catch (Exception $e) {
        return ['can_cancel' => false, 'message' => 'Error checking bid'];
    }
}

/**
 * Cancel Bid and Refund
 */
function cancelBid($pdo, $bidId, $userId) {
    $check = canCancelBid($pdo, $bidId, $userId);
    
    if (!$check['can_cancel']) {
        return ['success' => false, 'message' => $check['message']];
    }
    
    $bid = $check['bid'];
    
    $pdo->beginTransaction();
    try {
        // Mark bid as cancelled
        $stmt = $pdo->prepare("UPDATE bids SET is_cancelled = 1, cancelled_at = NOW(), status = 'lost' WHERE id = ?");
        $stmt->execute([$bidId]);
        
        // Get user info
        $stmt = $pdo->prepare("SELECT name, mobile FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $user = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // Refund amount
        $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id) VALUES (?, ?, ?, 'credit', ?, ?, 'bid_cancel', ?)");
        $stmt->execute([$userId, $user['name'], $user['mobile'], $bid['amount'], 'Bid Cancelled - Refund', $bidId]);
        
        // Update daily bid total
        $stmt = $pdo->prepare("UPDATE users SET daily_bid_total = GREATEST(0, daily_bid_total - ?) WHERE id = ? AND daily_bid_date = CURDATE()");
        $stmt->execute([$bid['amount'], $userId]);
        
        // Add notification
        addNotification($pdo, $userId, 'bid_cancelled', 'Bid Cancelled', 
            "Your bid of ₹{$bid['amount']} on {$bid['game_name_archive']} has been cancelled and refunded.");
        
        $pdo->commit();
        return ['success' => true, 'message' => 'Bid cancelled! ₹' . number_format($bid['amount'], 0) . ' refunded to wallet'];
    } catch (Exception $e) {
        $pdo->rollBack();
        return ['success' => false, 'message' => 'Error cancelling bid'];
    }
}

// =============================================
// NOTIFICATIONS
// =============================================

/**
 * Add Notification
 */
function addNotification($pdo, $userId, $type, $title, $message, $data = null) {
    try {
        $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, data) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$userId, $type, $title, $message, $data ? json_encode($data) : null]);
        return true;
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Get User Notifications
 */
function getNotifications($pdo, $userId, $limit = 20, $unreadOnly = false) {
    try {
        $where = $unreadOnly ? "AND is_read = 0" : "";
        $stmt = $pdo->prepare("SELECT * FROM notifications WHERE user_id = ? $where ORDER BY created_at DESC LIMIT ?");
        $stmt->execute([$userId, $limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return [];
    }
}

/**
 * Get Unread Count
 */
function getUnreadNotificationCount($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM notifications WHERE user_id = ? AND is_read = 0");
        $stmt->execute([$userId]);
        return $stmt->fetchColumn();
    } catch (Exception $e) {
        return 0;
    }
}

/**
 * Mark Notifications as Read
 */
function markNotificationsRead($pdo, $userId, $notificationId = null) {
    try {
        if ($notificationId) {
            $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
            $stmt->execute([$notificationId, $userId]);
        } else {
            $stmt = $pdo->prepare("UPDATE notifications SET is_read = 1 WHERE user_id = ?");
            $stmt->execute([$userId]);
        }
        return true;
    } catch (Exception $e) {
        return false;
    }
}

// =============================================
// REFERRAL SYSTEM
// =============================================

/**
 * Generate Unique Referral Code
 */
function generateReferralCode($userId) {
    return strtoupper(substr(md5(uniqid() . $userId), 0, 6) . $userId);
}

/**
 * Apply Referral Code
 */
function applyReferralCode($pdo, $newUserId, $referralCode) {
    try {
        // Find referrer
        $stmt = $pdo->prepare("SELECT id, name FROM users WHERE referral_code = ? AND is_admin = 0");
        $stmt->execute([$referralCode]);
        $referrer = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$referrer) {
            return ['success' => false, 'message' => 'Invalid referral code'];
        }
        
        if ($referrer['id'] == $newUserId) {
            return ['success' => false, 'message' => 'Cannot use your own referral code'];
        }
        
        // Check if already referred
        $stmt = $pdo->prepare("SELECT id FROM referrals WHERE referred_id = ?");
        $stmt->execute([$newUserId]);
        if ($stmt->fetch()) {
            return ['success' => false, 'message' => 'Referral already applied'];
        }
        
        // Get bonus amounts
        $referrerBonus = floatval(getSetting($pdo, 'referral_bonus_referrer', 50));
        $referredBonus = floatval(getSetting($pdo, 'referral_bonus_referred', 25));
        
        // Update new user's referred_by
        $stmt = $pdo->prepare("UPDATE users SET referred_by = ? WHERE id = ?");
        $stmt->execute([$referrer['id'], $newUserId]);
        
        // Create referral record
        $stmt = $pdo->prepare("INSERT INTO referrals (referrer_id, referred_id, referral_code, referrer_bonus, referred_bonus) VALUES (?, ?, ?, ?, ?)");
        $stmt->execute([$referrer['id'], $newUserId, $referralCode, $referrerBonus, $referredBonus]);
        
        return [
            'success' => true, 
            'message' => "Referral applied! You'll get ₹$referredBonus bonus on first deposit.",
            'referrer_name' => $referrer['name']
        ];
    } catch (Exception $e) {
        return ['success' => false, 'message' => 'Error applying referral'];
    }
}

/**
 * Process Referral Bonus (call after first deposit)
 */
function processReferralBonus($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("SELECT r.*, u.name as referrer_name, u.mobile as referrer_mobile, 
                              nu.name as new_user_name, nu.mobile as new_user_mobile
                              FROM referrals r 
                              JOIN users u ON r.referrer_id = u.id 
                              JOIN users nu ON r.referred_id = nu.id
                              WHERE r.referred_id = ? AND r.bonus_given = 0");
        $stmt->execute([$userId]);
        $referral = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$referral) {
            return false;
        }
        
        $pdo->beginTransaction();
        try {
            // Give bonus to referrer
            if ($referral['referrer_bonus'] > 0) {
                $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'credit', ?, ?)");
                $stmt->execute([$referral['referrer_id'], $referral['referrer_name'], $referral['referrer_mobile'], $referral['referrer_bonus'], 'Referral Bonus - ' . $referral['new_user_name']]);
                
                addNotification($pdo, $referral['referrer_id'], 'referral_bonus', 'Referral Bonus! 🎉', 
                    "You earned ₹{$referral['referrer_bonus']} for referring {$referral['new_user_name']}!");
            }
            
            // Give bonus to new user
            if ($referral['referred_bonus'] > 0) {
                $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'credit', ?, ?)");
                $stmt->execute([$referral['referred_id'], $referral['new_user_name'], $referral['new_user_mobile'], $referral['referred_bonus'], 'Welcome Bonus - Referral']);
                
                addNotification($pdo, $referral['referred_id'], 'welcome_bonus', 'Welcome Bonus! 🎉', 
                    "You received ₹{$referral['referred_bonus']} welcome bonus!");
            }
            
            // Mark bonus as given
            $stmt = $pdo->prepare("UPDATE referrals SET bonus_given = 1 WHERE id = ?");
            $stmt->execute([$referral['id']]);
            
            $pdo->commit();
            return true;
        } catch (Exception $e) {
            $pdo->rollBack();
            return false;
        }
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Get User's Referral Stats
 */
function getReferralStats($pdo, $userId) {
    try {
        $stmt = $pdo->prepare("SELECT referral_code FROM users WHERE id = ?");
        $stmt->execute([$userId]);
        $code = $stmt->fetchColumn();
        
        $stmt = $pdo->prepare("SELECT COUNT(*) as total_referrals, SUM(referrer_bonus) as total_earned FROM referrals WHERE referrer_id = ? AND bonus_given = 1");
        $stmt->execute([$userId]);
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        return [
            'referral_code' => $code,
            'total_referrals' => $stats['total_referrals'] ?? 0,
            'total_earned' => $stats['total_earned'] ?? 0
        ];
    } catch (Exception $e) {
        return ['referral_code' => '', 'total_referrals' => 0, 'total_earned' => 0];
    }
}

// =============================================
// ADMIN ROLE PERMISSIONS
// =============================================

/**
 * Check Admin Permission
 */
function hasPermission($pdo, $adminId, $permission) {
    try {
        $stmt = $pdo->prepare("SELECT ar.permissions FROM users u JOIN admin_roles ar ON u.admin_role_id = ar.id WHERE u.id = ? AND u.is_admin = 1");
        $stmt->execute([$adminId]);
        $permissions = $stmt->fetchColumn();
        
        if (!$permissions) {
            return false;
        }
        
        $perms = json_decode($permissions, true);
        
        // Super admin has all permissions
        if (isset($perms['all']) && $perms['all']) {
            return true;
        }
        
        return isset($perms[$permission]) && $perms[$permission];
    } catch (Exception $e) {
        return false;
    }
}

/**
 * Get Admin Role
 */
function getAdminRole($pdo, $adminId) {
    try {
        $stmt = $pdo->prepare("SELECT ar.* FROM users u JOIN admin_roles ar ON u.admin_role_id = ar.id WHERE u.id = ?");
        $stmt->execute([$adminId]);
        return $stmt->fetch(PDO::FETCH_ASSOC);
    } catch (Exception $e) {
        return null;
    }
}

/**
 * Check Permission and Redirect
 */
function requirePermission($pdo, $adminId, $permission, $redirectUrl = 'index.php') {
    if (!hasPermission($pdo, $adminId, $permission)) {
        header("Location: $redirectUrl?error=Permission+denied");
        exit();
    }
}
?>
