<?php
/**
 * Profile API - Get Profile, Update Profile, Change Password
 * Endpoints:
 *   GET  /api/profile.php?action=get
 *   POST /api/profile.php?action=update
 *   POST /api/profile.php?action=change_password
 *   POST /api/profile.php?action=upload_photo
 *   POST /api/profile.php?action=upload_qr
 */

require_once 'config.php';

$action = $_GET['action'] ?? 'get';

switch ($action) {
    case 'get':
        handleGetProfile($pdo);
        break;
    case 'update':
        handleUpdateProfile($pdo);
        break;
    case 'change_password':
        handleChangePassword($pdo);
        break;
    case 'upload_photo':
        handleUploadPhoto($pdo);
        break;
    case 'upload_qr':
        handleUploadQR($pdo);
        break;
    case 'stats':
        handleGetStats($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get User Profile
 */
function handleGetProfile($pdo) {
    $user = authenticate($pdo);
    
    $balance = getWalletBalance($pdo, $user['id']);
    
    // Get stats
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total_bids,
        COALESCE(SUM(amount), 0) as total_bid_amount,
        COUNT(CASE WHEN status = 'won' THEN 1 END) as total_wins,
        COALESCE(SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END), 0) as total_won
        FROM bids WHERE user_id = ? AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Get referral stats
    $stmt = $pdo->prepare("SELECT COUNT(*) as total_referrals FROM referrals WHERE referrer_id = ?");
    $stmt->execute([$user['id']]);
    $referralStats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendResponse([
        'user' => [
            'id' => (int)$user['id'],
            'name' => $user['name'],
            'mobile' => $user['mobile'],
            'status' => $user['status'],
            'balance' => (float)$balance,
            'referral_code' => $user['referral_code'],
            'has_pin' => !empty($user['transaction_pin']),
            'profile_photo' => $user['profile_photo'] ? '/uploads/' . $user['profile_photo'] : null,
            'qr_code' => $user['qr_code'] ? '/uploads/' . $user['qr_code'] : null,
            'created_at' => $user['created_at'],
            'last_login' => $user['last_login']
        ],
        'bank_details' => [
            'bank_name' => $user['bank_name'],
            'account_no' => $user['account_no'],
            'account_holder' => $user['account_holder'],
            'ifsc_code' => $user['ifsc_code'],
            'upi_id' => $user['upi_id']
        ],
        'stats' => [
            'total_bids' => (int)$stats['total_bids'],
            'total_bid_amount' => (float)$stats['total_bid_amount'],
            'total_wins' => (int)$stats['total_wins'],
            'total_won' => (float)$stats['total_won'],
            'total_referrals' => (int)$referralStats['total_referrals']
        ]
    ]);
}

/**
 * Update User Profile
 */
function handleUpdateProfile($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    // Fields that can be updated
    $allowedFields = ['name', 'bank_name', 'account_no', 'account_holder', 'ifsc_code', 'upi_id'];
    
    $updates = [];
    $params = [];
    
    foreach ($allowedFields as $field) {
        if (isset($data[$field])) {
            $updates[] = "{$field} = ?";
            $params[] = trim($data[$field]);
        }
    }
    
    if (empty($updates)) {
        sendError('No fields to update', 400);
    }
    
    // Validate name if provided
    if (isset($data['name']) && empty(trim($data['name']))) {
        sendError('Name cannot be empty', 400);
    }
    
    // Validate IFSC if provided
    if (isset($data['ifsc_code']) && !empty($data['ifsc_code'])) {
        if (!preg_match('/^[A-Z]{4}0[A-Z0-9]{6}$/', strtoupper($data['ifsc_code']))) {
            sendError('Invalid IFSC code format', 400);
        }
        // Find and update the IFSC to uppercase
        $key = array_search('ifsc_code = ?', $updates);
        if ($key !== false) {
            $params[$key] = strtoupper($data['ifsc_code']);
        }
    }
    
    $params[] = $user['id'];
    
    $sql = "UPDATE users SET " . implode(', ', $updates) . ", updated_at = NOW() WHERE id = ?";
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    
    // Get updated user
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$user['id']]);
    $updatedUser = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendResponse([
        'message' => 'Profile updated successfully',
        'user' => [
            'id' => (int)$updatedUser['id'],
            'name' => $updatedUser['name'],
            'mobile' => $updatedUser['mobile']
        ],
        'bank_details' => [
            'bank_name' => $updatedUser['bank_name'],
            'account_no' => $updatedUser['account_no'],
            'account_holder' => $updatedUser['account_holder'],
            'ifsc_code' => $updatedUser['ifsc_code'],
            'upi_id' => $updatedUser['upi_id']
        ]
    ]);
}

/**
 * Change Password
 */
function handleChangePassword($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['current_password', 'new_password']);
    
    $currentPassword = $data['current_password'];
    $newPassword = $data['new_password'];
    $confirmPassword = $data['confirm_password'] ?? $newPassword;
    
    // Verify current password
    if (!password_verify($currentPassword, $user['password'])) {
        sendError('Current password is incorrect', 401);
    }
    
    // Validate new password
    if (strlen($newPassword) < 6) {
        sendError('New password must be at least 6 characters', 400);
    }
    
    if ($newPassword !== $confirmPassword) {
        sendError('Passwords do not match', 400);
    }
    
    // Update password
    $hashedPassword = password_hash($newPassword, PASSWORD_DEFAULT);
    $stmt = $pdo->prepare("UPDATE users SET password = ?, updated_at = NOW() WHERE id = ?");
    $stmt->execute([$hashedPassword, $user['id']]);
    
    sendResponse([
        'message' => 'Password changed successfully'
    ]);
}

/**
 * Upload Profile Photo
 */
function handleUploadPhoto($pdo) {
    $user = authenticate($pdo);
    
    if (!isset($_FILES['photo']) || $_FILES['photo']['error'] !== 0) {
        sendError('Profile photo required', 400);
    }
    
    $file = $_FILES['photo'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowed = ['jpg', 'jpeg', 'png', 'gif'];
    
    if (!in_array($ext, $allowed)) {
        sendError('Invalid file type. Use JPG, PNG or GIF.', 400);
    }
    
    if ($file['size'] > 2 * 1024 * 1024) {
        sendError('File too large. Max 2MB allowed.', 400);
    }
    
    // Delete old photo
    if ($user['profile_photo']) {
        $oldPath = __DIR__ . '/../uploads/' . $user['profile_photo'];
        if (file_exists($oldPath)) {
            unlink($oldPath);
        }
    }
    
    $filename = 'profile_' . $user['id'] . '_' . time() . '.' . $ext;
    $uploadPath = __DIR__ . '/../uploads/' . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        sendError('Failed to upload photo', 500);
    }
    
    // Update database
    $stmt = $pdo->prepare("UPDATE users SET profile_photo = ? WHERE id = ?");
    $stmt->execute([$filename, $user['id']]);
    
    sendResponse([
        'message' => 'Profile photo updated',
        'photo_url' => '/uploads/' . $filename
    ]);
}

/**
 * Upload QR Code
 */
function handleUploadQR($pdo) {
    $user = authenticate($pdo);
    
    if (!isset($_FILES['qr_code']) || $_FILES['qr_code']['error'] !== 0) {
        sendError('QR code image required', 400);
    }
    
    $file = $_FILES['qr_code'];
    $ext = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowed = ['jpg', 'jpeg', 'png', 'gif'];
    
    if (!in_array($ext, $allowed)) {
        sendError('Invalid file type. Use JPG, PNG or GIF.', 400);
    }
    
    if ($file['size'] > 2 * 1024 * 1024) {
        sendError('File too large. Max 2MB allowed.', 400);
    }
    
    // Delete old QR
    if ($user['qr_code']) {
        $oldPath = __DIR__ . '/../uploads/' . $user['qr_code'];
        if (file_exists($oldPath)) {
            unlink($oldPath);
        }
    }
    
    $filename = 'qr_' . $user['id'] . '_' . time() . '.' . $ext;
    $uploadPath = __DIR__ . '/../uploads/' . $filename;
    
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        sendError('Failed to upload QR code', 500);
    }
    
    // Update database
    $stmt = $pdo->prepare("UPDATE users SET qr_code = ? WHERE id = ?");
    $stmt->execute([$filename, $user['id']]);
    
    sendResponse([
        'message' => 'QR code updated',
        'qr_url' => '/uploads/' . $filename
    ]);
}

/**
 * Get User Stats
 */
function handleGetStats($pdo) {
    $user = authenticate($pdo);
    
    // Overall stats
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total_bids,
        COALESCE(SUM(amount), 0) as total_bid_amount,
        COUNT(CASE WHEN status = 'won' THEN 1 END) as total_wins,
        COUNT(CASE WHEN status = 'lost' THEN 1 END) as total_losses,
        COALESCE(SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END), 0) as total_won
        FROM bids WHERE user_id = ? AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $overall = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Today's stats
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as today_bids,
        COALESCE(SUM(amount), 0) as today_amount,
        COALESCE(SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END), 0) as today_won
        FROM bids WHERE user_id = ? AND DATE(created_at) = CURDATE() AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $today = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // This month stats
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as month_bids,
        COALESCE(SUM(amount), 0) as month_amount,
        COALESCE(SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END), 0) as month_won
        FROM bids WHERE user_id = ? AND MONTH(created_at) = MONTH(CURDATE()) AND YEAR(created_at) = YEAR(CURDATE()) AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $month = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Wallet stats
    $stmt = $pdo->prepare("SELECT 
        COALESCE(SUM(CASE WHEN type = 'deposit' AND status = 'approved' THEN amount ELSE 0 END), 0) as total_deposits,
        COALESCE(SUM(CASE WHEN type = 'withdraw' AND status = 'approved' THEN amount ELSE 0 END), 0) as total_withdrawals
        FROM wallet_requests WHERE user_id = ?");
    $stmt->execute([$user['id']]);
    $wallet = $stmt->fetch(PDO::FETCH_ASSOC);
    
    // Win rate
    $winRate = $overall['total_bids'] > 0 
        ? round(($overall['total_wins'] / $overall['total_bids']) * 100, 1) 
        : 0;
    
    // Profit/Loss
    $profitLoss = $overall['total_won'] - $overall['total_bid_amount'];
    
    sendResponse([
        'overall' => [
            'total_bids' => (int)$overall['total_bids'],
            'total_bid_amount' => (float)$overall['total_bid_amount'],
            'total_wins' => (int)$overall['total_wins'],
            'total_losses' => (int)$overall['total_losses'],
            'total_won' => (float)$overall['total_won'],
            'win_rate' => $winRate,
            'profit_loss' => (float)$profitLoss
        ],
        'today' => [
            'bids' => (int)$today['today_bids'],
            'amount' => (float)$today['today_amount'],
            'won' => (float)$today['today_won']
        ],
        'this_month' => [
            'bids' => (int)$month['month_bids'],
            'amount' => (float)$month['month_amount'],
            'won' => (float)$month['month_won']
        ],
        'wallet' => [
            'total_deposits' => (float)$wallet['total_deposits'],
            'total_withdrawals' => (float)$wallet['total_withdrawals']
        ]
    ]);
}
?>
