<?php
/**
 * Bids API - Place bids, Bid history, Cancel bid
 * Endpoints:
 *   POST /api/bids.php?action=place
 *   GET  /api/bids.php?action=history
 *   POST /api/bids.php?action=cancel
 *   GET  /api/bids.php?action=today
 */

require_once 'config.php';
require_once __DIR__ . '/../config/security.php';

$action = $_GET['action'] ?? '';

switch ($action) {
    case 'place':
        handlePlaceBid($pdo);
        break;
    case 'history':
        handleBidHistory($pdo);
        break;
    case 'cancel':
        handleCancelBid($pdo);
        break;
    case 'today':
        handleTodayBids($pdo);
        break;
    case 'game_bids':
        handleGameBids($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Place Bid(s)
 */
function handlePlaceBid($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['game_id', 'bids']);
    
    $gameId = intval($data['game_id']);
    $bids = $data['bids']; // Array of {type, number, amount}
    
    if (!is_array($bids) || empty($bids)) {
        sendError('At least one bid is required', 400);
    }
    
    // Get game
    $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ? AND status = 'active'");
    $stmt->execute([$gameId]);
    $game = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$game) {
        sendError('Game not found', 404);
    }
    
    // Check if game is open
    if (!isGameOpenForBid($pdo, $game)) {
        sendError('Game is closed for bidding', 400);
    }
    
    $playingDate = getPlayingDateForBid($game);
    
    // Validate and calculate total
    $totalAmount = 0;
    $validBids = [];
    $rate = (int)getSetting($pdo, 'jodi_rate', '95');
    $minBid = (int)getSetting($pdo, 'min_bid', '10');
    $maxBid = (int)getSetting($pdo, 'max_bid', '10000');
    
    foreach ($bids as $bid) {
        $type = strtolower($bid['type'] ?? '');
        $number = $bid['number'] ?? '';
        $amount = floatval($bid['amount'] ?? 0);
        
        // Validate bid type
        if (!in_array($type, ['jodi', 'andar', 'bahar'])) {
            sendError("Invalid bid type: {$type}", 400);
        }
        
        // Validate number
        if ($type === 'jodi') {
            if (!preg_match('/^[0-9]{2}$/', $number)) {
                sendError('Jodi must be 2 digits (00-99)', 400);
            }
        } else {
            if (!preg_match('/^[0-9]$/', $number)) {
                sendError('Andar/Bahar must be single digit (0-9)', 400);
            }
        }
        
        // Validate amount
        if ($amount < $minBid) {
            sendError("Minimum bid amount is ₹{$minBid}", 400);
        }
        if ($amount > $maxBid) {
            sendError("Maximum bid amount is ₹{$maxBid}", 400);
        }
        
        $totalAmount += $amount;
        $validBids[] = [
            'type' => $type,
            'number' => str_pad($number, $type === 'jodi' ? 2 : 1, '0', STR_PAD_LEFT),
            'amount' => $amount
        ];
    }
    
    // Check daily bid limit
    $dailyLimit = (int)getSetting($pdo, 'daily_bid_limit', '100000');
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(amount), 0) as total FROM bids WHERE user_id = ? AND DATE(created_at) = CURDATE() AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $todayTotal = $stmt->fetchColumn();
    
    if ($todayTotal + $totalAmount > $dailyLimit) {
        $remaining = $dailyLimit - $todayTotal;
        sendError("Daily bid limit exceeded. Remaining: ₹" . number_format($remaining, 0), 400);
    }
    
    // Check wallet balance
    $balance = getWalletBalance($pdo, $user['id']);
    if ($totalAmount > $balance) {
        sendError('Insufficient balance. Need ₹' . number_format($totalAmount, 2), 400);
    }
    
    // Rate limiting - max 50 bids per minute
    if (!checkApiRateLimit($pdo, 'bid_' . $user['id'], 50, 60)) {
        sendError('Too many bids. Please wait a moment.', 429);
    }
    
    // Place bids
    $pdo->beginTransaction();
    try {
        $placedBids = [];
        
        foreach ($validBids as $bid) {
            // Insert bid
            $stmt = $pdo->prepare("INSERT INTO bids (user_id, user_name_archive, user_mobile_archive, game_id, game_name_archive, bid_type, number, amount, playing_date, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'pending', NOW())");
            $stmt->execute([
                $user['id'],
                $user['name'],
                $user['mobile'],
                $gameId,
                $game['name'],
                $bid['type'],
                $bid['number'],
                $bid['amount'],
                $playingDate
            ]);
            $bidId = $pdo->lastInsertId();
            
            // Deduct from wallet
            $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id, created_at) VALUES (?, ?, ?, 'debit', ?, ?, 'bid', ?, NOW())");
            $stmt->execute([
                $user['id'],
                $user['name'],
                $user['mobile'],
                $bid['amount'],
                "Bid {$bid['type']} {$bid['number']} - {$game['name']}",
                $bidId
            ]);
            
            $potentialWin = $bid['type'] === 'jodi' ? $bid['amount'] * $rate : $bid['amount'] * ($rate / 10);
            
            $placedBids[] = [
                'bid_id' => (int)$bidId,
                'type' => $bid['type'],
                'number' => $bid['number'],
                'amount' => $bid['amount'],
                'potential_win' => round($potentialWin, 2)
            ];
        }
        
        // Update daily bid total
        $stmt = $pdo->prepare("UPDATE users SET daily_bid_total = COALESCE(daily_bid_total, 0) + ?, daily_bid_date = CURDATE() WHERE id = ?");
        $stmt->execute([$totalAmount, $user['id']]);
        
        $pdo->commit();
        
        // Get new balance
        $newBalance = getWalletBalance($pdo, $user['id']);
        
        sendResponse([
            'message' => count($placedBids) . ' bid(s) placed successfully!',
            'bids' => $placedBids,
            'total_amount' => $totalAmount,
            'new_balance' => (float)$newBalance,
            'game' => [
                'id' => (int)$gameId,
                'name' => $game['name'],
                'playing_date' => $playingDate
            ]
        ], 201);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        sendError('Failed to place bid. Please try again.', 500);
    }
}

/**
 * Get Bid History
 */
function handleBidHistory($pdo) {
    $user = authenticate($pdo);
    
    $status = $_GET['status'] ?? ''; // pending, won, lost, cancelled
    $gameId = intval($_GET['game_id'] ?? 0);
    $limit = min(intval($_GET['limit'] ?? 50), 100);
    $page = max(intval($_GET['page'] ?? 1), 1);
    $offset = ($page - 1) * $limit;
    
    // Build query
    $where = ["b.user_id = ?"];
    $params = [$user['id']];
    
    if ($status && in_array($status, ['pending', 'won', 'lost', 'cancelled'])) {
        if ($status === 'cancelled') {
            $where[] = "b.is_cancelled = 1";
        } else {
            $where[] = "b.status = ? AND b.is_cancelled = 0";
            $params[] = $status;
        }
    }
    
    if ($gameId) {
        $where[] = "b.game_id = ?";
        $params[] = $gameId;
    }
    
    $whereClause = implode(' AND ', $where);
    
    // Get total count
    $stmt = $pdo->prepare("SELECT COUNT(*) FROM bids b WHERE {$whereClause}");
    $stmt->execute($params);
    $total = $stmt->fetchColumn();
    
    // Get bids
    $params[] = $limit;
    $params[] = $offset;
    $stmt = $pdo->prepare("
        SELECT b.*, g.name as game_name 
        FROM bids b 
        LEFT JOIN games g ON b.game_id = g.id 
        WHERE {$whereClause} 
        ORDER BY b.created_at DESC 
        LIMIT ? OFFSET ?
    ");
    $stmt->execute($params);
    $bids = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $bidsList = [];
    foreach ($bids as $bid) {
        $bidsList[] = [
            'id' => (int)$bid['id'],
            'game_id' => (int)$bid['game_id'],
            'game_name' => $bid['game_name'] ?? $bid['game_name_archive'],
            'bid_type' => $bid['bid_type'],
            'number' => $bid['number'],
            'amount' => (float)$bid['amount'],
            'status' => $bid['is_cancelled'] ? 'cancelled' : $bid['status'],
            'win_amount' => (float)($bid['win_amount'] ?? 0),
            'playing_date' => $bid['playing_date'],
            'created_at' => $bid['created_at'],
            'is_cancelled' => (bool)$bid['is_cancelled'],
            'cancelled_at' => $bid['cancelled_at']
        ];
    }
    
    // Calculate totals
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total_bids,
        COALESCE(SUM(amount), 0) as total_amount,
        COALESCE(SUM(CASE WHEN status = 'won' THEN win_amount ELSE 0 END), 0) as total_won
        FROM bids WHERE user_id = ? AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendResponse([
        'bids' => $bidsList,
        'stats' => [
            'total_bids' => (int)$stats['total_bids'],
            'total_amount' => (float)$stats['total_amount'],
            'total_won' => (float)$stats['total_won']
        ],
        'pagination' => [
            'current_page' => $page,
            'per_page' => $limit,
            'total' => (int)$total,
            'total_pages' => ceil($total / $limit)
        ]
    ]);
}

/**
 * Cancel Bid
 */
function handleCancelBid($pdo) {
    $user = authenticate($pdo);
    $data = getJsonInput();
    
    validateRequired($data, ['bid_id']);
    
    $bidId = intval($data['bid_id']);
    
    // Get bid
    $stmt = $pdo->prepare("SELECT b.*, g.close_time, g.is_overnight FROM bids b JOIN games g ON b.game_id = g.id WHERE b.id = ? AND b.user_id = ?");
    $stmt->execute([$bidId, $user['id']]);
    $bid = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$bid) {
        sendError('Bid not found', 404);
    }
    
    if ($bid['status'] !== 'pending') {
        sendError('Only pending bids can be cancelled', 400);
    }
    
    if ($bid['is_cancelled']) {
        sendError('Bid is already cancelled', 400);
    }
    
    // Check cancellation time window (5 minutes before close)
    $cancelMinutes = (int)getSetting($pdo, 'bid_cancel_minutes', '5');
    $now = new DateTime();
    $closeTime = new DateTime($bid['playing_date'] . ' ' . $bid['close_time']);
    
    if ($bid['is_overnight']) {
        $closeTime->modify('+1 day');
    }
    
    $closeTime->modify("-{$cancelMinutes} minutes");
    
    if ($now >= $closeTime) {
        sendError("Cannot cancel bid within {$cancelMinutes} minutes of close time", 400);
    }
    
    // Cancel bid and refund
    $pdo->beginTransaction();
    try {
        // Mark bid as cancelled
        $stmt = $pdo->prepare("UPDATE bids SET is_cancelled = 1, cancelled_at = NOW(), refund_processed = 1 WHERE id = ?");
        $stmt->execute([$bidId]);
        
        // Refund to wallet
        $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description, reference_type, reference_id, created_at) VALUES (?, ?, ?, 'credit', ?, ?, 'bid_cancel', ?, NOW())");
        $stmt->execute([
            $user['id'],
            $user['name'],
            $user['mobile'],
            $bid['amount'],
            "Bid Cancelled - Refund",
            $bidId
        ]);
        
        // Update daily bid total
        $stmt = $pdo->prepare("UPDATE users SET daily_bid_total = GREATEST(0, COALESCE(daily_bid_total, 0) - ?) WHERE id = ?");
        $stmt->execute([$bid['amount'], $user['id']]);
        
        // Add notification
        $stmt = $pdo->prepare("INSERT INTO notifications (user_id, type, title, message, created_at) VALUES (?, 'bid_cancelled', 'Bid Cancelled 🔄', ?, NOW())");
        $stmt->execute([
            $user['id'],
            "Your bid of ₹{$bid['amount']} on {$bid['bid_type']} {$bid['number']} has been cancelled and refunded."
        ]);
        
        $pdo->commit();
        
        $newBalance = getWalletBalance($pdo, $user['id']);
        
        sendResponse([
            'message' => 'Bid cancelled and refunded successfully',
            'refund_amount' => (float)$bid['amount'],
            'new_balance' => (float)$newBalance
        ]);
        
    } catch (Exception $e) {
        $pdo->rollBack();
        sendError('Failed to cancel bid', 500);
    }
}

/**
 * Get Today's Bids
 */
function handleTodayBids($pdo) {
    $user = authenticate($pdo);
    
    $stmt = $pdo->prepare("
        SELECT b.*, g.name as game_name, g.close_time, g.is_overnight
        FROM bids b 
        JOIN games g ON b.game_id = g.id 
        WHERE b.user_id = ? AND b.playing_date = CURDATE() AND b.is_cancelled = 0
        ORDER BY b.created_at DESC
    ");
    $stmt->execute([$user['id']]);
    $bids = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $cancelMinutes = (int)getSetting($pdo, 'bid_cancel_minutes', '5');
    $now = new DateTime();
    
    $bidsList = [];
    foreach ($bids as $bid) {
        $closeTime = new DateTime($bid['playing_date'] . ' ' . $bid['close_time']);
        if ($bid['is_overnight']) {
            $closeTime->modify('+1 day');
        }
        $closeTime->modify("-{$cancelMinutes} minutes");
        
        $canCancel = ($bid['status'] === 'pending') && ($now < $closeTime);
        
        $bidsList[] = [
            'id' => (int)$bid['id'],
            'game_name' => $bid['game_name'],
            'bid_type' => $bid['bid_type'],
            'number' => $bid['number'],
            'amount' => (float)$bid['amount'],
            'status' => $bid['status'],
            'win_amount' => (float)($bid['win_amount'] ?? 0),
            'can_cancel' => $canCancel,
            'created_at' => $bid['created_at']
        ];
    }
    
    // Today's totals
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total_bids,
        COALESCE(SUM(amount), 0) as total_amount
        FROM bids WHERE user_id = ? AND playing_date = CURDATE() AND is_cancelled = 0");
    $stmt->execute([$user['id']]);
    $totals = $stmt->fetch(PDO::FETCH_ASSOC);
    
    sendResponse([
        'bids' => $bidsList,
        'totals' => [
            'count' => (int)$totals['total_bids'],
            'amount' => (float)$totals['total_amount']
        ],
        'date' => date('Y-m-d')
    ]);
}

/**
 * Get bids for specific game
 */
function handleGameBids($pdo) {
    $user = authenticate($pdo);
    $gameId = intval($_GET['game_id'] ?? 0);
    $date = $_GET['date'] ?? date('Y-m-d');
    
    if (!$gameId) {
        sendError('Game ID required', 400);
    }
    
    $stmt = $pdo->prepare("
        SELECT * FROM bids 
        WHERE user_id = ? AND game_id = ? AND playing_date = ? AND is_cancelled = 0
        ORDER BY created_at DESC
    ");
    $stmt->execute([$user['id'], $gameId, $date]);
    $bids = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $bidsList = [];
    foreach ($bids as $bid) {
        $bidsList[] = [
            'id' => (int)$bid['id'],
            'bid_type' => $bid['bid_type'],
            'number' => $bid['number'],
            'amount' => (float)$bid['amount'],
            'status' => $bid['status'],
            'win_amount' => (float)($bid['win_amount'] ?? 0),
            'created_at' => $bid['created_at']
        ];
    }
    
    sendResponse([
        'bids' => $bidsList,
        'date' => $date
    ]);
}

/**
 * Helper: Check if game is open for bidding
 */
function isGameOpenForBid($pdo, $game) {
    $now = new DateTime();
    $currentTime = $now->format('H:i:s');
    
    $openTime = $game['open_time'];
    $closeTime = $game['close_time'];
    $isOvernight = $game['is_overnight'];
    
    // Check if result already published
    $playingDate = getPlayingDateForBid($game);
    $stmt = $pdo->prepare("SELECT id FROM results WHERE game_id = ? AND playing_date = ?");
    $stmt->execute([$game['id'], $playingDate]);
    if ($stmt->fetch()) {
        return false;
    }
    
    if ($isOvernight) {
        if ($currentTime >= $openTime || $currentTime < $closeTime) {
            return true;
        }
    } else {
        if ($currentTime >= $openTime && $currentTime < $closeTime) {
            return true;
        }
    }
    
    return false;
}

/**
 * Helper: Get playing date for bid
 */
function getPlayingDateForBid($game) {
    $now = new DateTime();
    $currentTime = $now->format('H:i:s');
    
    if ($game['is_overnight']) {
        if ($currentTime < $game['close_time']) {
            $now->modify('-1 day');
        }
    }
    
    return $now->format('Y-m-d');
}
?>
