<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);

// Track admin session
trackUserSession($pdo, $admin['id']);

// Activity Log Function

// Calculate User Stats
function getUserStats($pdo, $user_id) {
    $stats = [];
    
    // Balance
    $stmt = $pdo->prepare("SELECT 
        COALESCE(SUM(CASE WHEN type='credit' THEN amount ELSE 0 END), 0) as total_credit,
        COALESCE(SUM(CASE WHEN type='debit' THEN amount ELSE 0 END), 0) as total_debit
        FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $wallet = $stmt->fetch();
    $stats['balance'] = $wallet['total_credit'] - $wallet['total_debit'];
    $stats['total_credit'] = $wallet['total_credit'];
    $stats['total_debit'] = $wallet['total_debit'];
    
    // Bids
    $stmt = $pdo->prepare("SELECT 
        COUNT(*) as total_bids,
        COALESCE(SUM(amount), 0) as total_bid_amount,
        SUM(CASE WHEN status='won' THEN 1 ELSE 0 END) as total_wins,
        COALESCE(SUM(CASE WHEN status='won' THEN win_amount ELSE 0 END), 0) as total_win_amount,
        SUM(CASE WHEN status='pending' THEN 1 ELSE 0 END) as pending_bids
        FROM bids WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $bids = $stmt->fetch();
    $stats = array_merge($stats, $bids);
    
    // Deposits & Withdrawals (Approved)
    $stmt = $pdo->prepare("SELECT 
        COALESCE(SUM(CASE WHEN type='deposit' AND status='approved' THEN amount ELSE 0 END), 0) as total_deposits,
        COALESCE(SUM(CASE WHEN type='withdraw' AND status='approved' THEN amount ELSE 0 END), 0) as total_withdrawals,
        SUM(CASE WHEN status='pending' THEN 1 ELSE 0 END) as pending_requests
        FROM wallet_requests WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $requests = $stmt->fetch();
    $stats = array_merge($stats, $requests);
    
    // Wallet transactions count
    $stmt = $pdo->prepare("SELECT COUNT(*) as wallet_txns FROM wallet WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $stats['wallet_txns'] = $stmt->fetch()['wallet_txns'];
    
    // Wallet requests count
    $stmt = $pdo->prepare("SELECT COUNT(*) as wallet_requests FROM wallet_requests WHERE user_id = ?");
    $stmt->execute([$user_id]);
    $stats['wallet_requests'] = $stmt->fetch()['wallet_requests'];
    
    return $stats;
}

// Archive and Delete User
function archiveAndDeleteUser($pdo, $user_id, $admin_id) {
    // Get user data
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_admin = 0");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$user) {
        return ['success' => false, 'message' => 'User not found or is admin'];
    }
    
    // Get admin name for archive
    $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
    $stmt->execute([$admin_id]);
    $admin_name = $stmt->fetchColumn() ?: 'Admin';
    
    // Get user stats
    $stats = getUserStats($pdo, $user_id);
    
    // Check for pending items
    if ($stats['pending_bids'] > 0) {
        return ['success' => false, 'message' => "Cannot delete! User has {$stats['pending_bids']} pending bids. Please publish results first."];
    }
    
    if ($stats['pending_requests'] > 0) {
        return ['success' => false, 'message' => "Cannot delete! User has {$stats['pending_requests']} pending wallet requests. Please approve/reject them first."];
    }
    
    $pdo->beginTransaction();
    
    try {
        // 1. Archive User (matching actual table structure)
        $stmt = $pdo->prepare("INSERT INTO users_archive 
            (id, name, mobile, password, upi_id, bank_name, account_no, ifsc_code, account_holder, is_admin, original_status, 
            original_created_at, deleted_by, deleted_by_name,
            final_balance, total_bids, total_bid_amount, total_wins, total_win_amount, total_deposits, total_withdrawals)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $user['id'], 
            $user['name'], 
            $user['mobile'], 
            $user['password'],
            $user['upi_id'] ?? null, 
            $user['bank_name'] ?? null, 
            $user['account_no'] ?? $user['account_number'] ?? null, 
            $user['ifsc_code'] ?? null,
            $user['account_holder'] ?? $user['name'],
            $user['is_admin'], 
            $user['status'], 
            $user['created_at'] ?? date('Y-m-d H:i:s'), 
            $admin_id,
            $admin_name,
            $stats['balance'], 
            $stats['total_bids'], 
            $stats['total_bid_amount'], 
            $stats['total_wins'], 
            $stats['total_win_amount'], 
            $stats['total_deposits'], 
            $stats['total_withdrawals']
        ]);
        
        // 2. Archive Bids
        $stmt = $pdo->prepare("INSERT INTO bids_archive 
            (id, user_id, user_name, user_mobile, game_id, game_name, bid_type, number, amount, win_amount, status, playing_date, original_created_at)
            SELECT b.id, b.user_id, ?, ?, b.game_id, 
            COALESCE(g.name, b.game_name_archive, CONCAT('Game #', b.game_id)),
            b.bid_type, b.number, b.amount, b.win_amount, b.status, b.playing_date, b.created_at
            FROM bids b 
            LEFT JOIN games g ON b.game_id = g.id
            WHERE b.user_id = ?");
        $stmt->execute([$user['name'], $user['mobile'], $user_id]);
        $archivedBids = $stmt->rowCount();
        
        // 3. Archive Wallet Transactions
        $stmt = $pdo->prepare("INSERT INTO wallet_archive 
            (id, user_id, user_name, user_mobile, type, amount, description, original_created_at)
            SELECT id, user_id, ?, ?, type, amount, description, created_at
            FROM wallet WHERE user_id = ?");
        $stmt->execute([$user['name'], $user['mobile'], $user_id]);
        $archivedWallet = $stmt->rowCount();
        
        // 4. Archive Wallet Requests
        $stmt = $pdo->prepare("INSERT INTO wallet_requests_archive 
            (id, user_id, user_name, user_mobile, type, amount, upi_id, screenshot, status, original_created_at)
            SELECT id, user_id, ?, ?, type, amount, upi_id, screenshot, status, created_at
            FROM wallet_requests WHERE user_id = ?");
        $stmt->execute([$user['name'], $user['mobile'], $user_id]);
        $archivedRequests = $stmt->rowCount();
        
        // 5. Delete from main tables
        $stmt = $pdo->prepare("DELETE FROM bids WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        $stmt = $pdo->prepare("DELETE FROM wallet WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        $stmt = $pdo->prepare("DELETE FROM wallet_requests WHERE user_id = ?");
        $stmt->execute([$user_id]);
        
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        
        // 6. Create Deletion Log
        $deletionSummary = [
            'user' => ['name' => $user['name'], 'mobile' => $user['mobile']],
            'stats' => $stats,
            'archived' => [
                'bids' => $archivedBids,
                'wallet_txns' => $archivedWallet,
                'wallet_requests' => $archivedRequests
            ]
        ];
        
        $stmt = $pdo->prepare("INSERT INTO deletion_log (entity_type, entity_id, entity_name, deleted_by, deleted_by_name, deletion_summary) VALUES (?, ?, ?, ?, ?, ?)");
        $stmt->execute(['user', $user_id, $user['name'], $admin_id, $admin_name, json_encode($deletionSummary)]);
        
        // 7. Activity Log
        logActivity($pdo, $admin_id, 'DELETE_USER_COMPLETE', 'users', $user_id, $user, $deletionSummary,
            "User {$user['name']} ({$user['mobile']}) deleted. Archived: {$archivedBids} bids, {$archivedWallet} wallet txns, {$archivedRequests} requests. Final balance: ₹{$stats['balance']}"
        );
        
        $pdo->commit();
        
        return [
            'success' => true, 
            'message' => "User deleted successfully! Archived: {$archivedBids} bids, {$archivedWallet} transactions, {$archivedRequests} requests.",
            'stats' => $stats
        ];
        
    } catch (Exception $e) {
        $pdo->rollBack();
        return ['success' => false, 'message' => 'Error: ' . $e->getMessage()];
    }
}

$error = '';
$success = '';

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    
    // Approve User (client-side captcha only)
    if (isset($_POST['approve_user'])) {
        $user_id = $_POST['user_id'];
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'APPROVE_USER', 'users', $user_id,
                ['status' => 'pending'], ['status' => 'active'],
                "User {$userData['name']} ({$userData['mobile']}) approved"
            );
            
            header("Location: users.php?msg=User+approved+successfully");
            exit();
        }
    }
    
    // Reject User (client-side captcha only)
    if (isset($_POST['reject_user'])) {
        $user_id = $_POST['user_id'];
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET status = 'rejected' WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'REJECT_USER', 'users', $user_id,
                ['status' => 'pending'], ['status' => 'rejected'],
                "User {$userData['name']} ({$userData['mobile']}) rejected"
            );
            
            header("Location: users.php?msg=User+rejected");
            exit();
        }
    }
    
    // Block User (client-side captcha only)
    if (isset($_POST['block_user'])) {
        $user_id = $_POST['user_id'];
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET status = 'blocked' WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'BLOCK_USER', 'users', $user_id,
                ['status' => $userData['status']], ['status' => 'blocked'],
                "User {$userData['name']} ({$userData['mobile']}) blocked"
            );
            
            header("Location: users.php?msg=User+blocked");
            exit();
        }
    }
    
    // Unblock User (client-side captcha only)
    if (isset($_POST['unblock_user'])) {
        $user_id = $_POST['user_id'];
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET status = 'active' WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'UNBLOCK_USER', 'users', $user_id,
                ['status' => 'blocked'], ['status' => 'active'],
                "User {$userData['name']} ({$userData['mobile']}) unblocked"
            );
            
            header("Location: users.php?msg=User+unblocked");
            exit();
        }
    }
    
    // Delete User (SERVER-SIDE captcha required - critical action)
    if (isset($_POST['confirm_delete_user'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $user_id = $_POST['user_id'];
            $confirm_text = trim($_POST['confirm_text'] ?? '');
            
            $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $userData = $stmt->fetch();
            
            if ($userData) {
                $expected = strtoupper($userData['mobile']);
                if ($confirm_text !== $expected) {
                    $error = "Confirmation failed! Please enter the mobile number correctly.";
                } else {
                    $result = archiveAndDeleteUser($pdo, $user_id, $admin['id']);
                    if ($result['success']) {
                        header("Location: users.php?msg=" . urlencode($result['message']));
                        exit();
                    } else {
                        $error = $result['message'];
                    }
                }
            }
        }
    }
    
    // Reset User Password (Admin sets new password)
    if (isset($_POST['reset_password'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '', 'password')) {
            $error = getCaptchaError();
        } else {
            $user_id = intval($_POST['user_id']);
            $new_password = trim($_POST['new_password'] ?? '');
            
            if (strlen($new_password) < 6) {
                $error = "Password must be at least 6 characters!";
            } else {
                $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_admin = 0");
                $stmt->execute([$user_id]);
                $userData = $stmt->fetch();
                
                if ($userData) {
                    $hashedPassword = password_hash($new_password, PASSWORD_DEFAULT);
                    $stmt = $pdo->prepare("UPDATE users SET password = ?, failed_login_attempts = 0, locked_until = NULL WHERE id = ?");
                    $stmt->execute([$hashedPassword, $user_id]);
                    
                    logActivity($pdo, $admin['id'], 'RESET_PASSWORD', 'users', $user_id,
                        [], ['password' => '***RESET***'],
                        "Password reset for {$userData['name']} ({$userData['mobile']})"
                    );
                    
                    header("Location: users.php?msg=Password+reset+successfully!+New+password:+" . urlencode($new_password));
                    exit();
                } else {
                    $error = "User not found!";
                }
            }
        }
    }
    
    // Reset User Transaction PIN
    if (isset($_POST['reset_pin'])) {
        $user_id = intval($_POST['user_id']);
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_admin = 0");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET transaction_pin = NULL WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'RESET_PIN', 'users', $user_id,
                [], ['transaction_pin' => 'CLEARED'],
                "Transaction PIN reset for {$userData['name']} ({$userData['mobile']})"
            );
            
            header("Location: users.php?msg=Transaction+PIN+reset+successfully!");
            exit();
        }
    }
    
    // Unlock User Account
    if (isset($_POST['unlock_account'])) {
        $user_id = intval($_POST['user_id']);
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_admin = 0");
        $stmt->execute([$user_id]);
        $userData = $stmt->fetch();
        
        if ($userData) {
            $stmt = $pdo->prepare("UPDATE users SET failed_login_attempts = 0, locked_until = NULL WHERE id = ?");
            $stmt->execute([$user_id]);
            
            logActivity($pdo, $admin['id'], 'UNLOCK_ACCOUNT', 'users', $user_id,
                ['locked_until' => $userData['locked_until']], ['locked_until' => null],
                "Account unlocked for {$userData['name']} ({$userData['mobile']})"
            );
            
            header("Location: users.php?msg=Account+unlocked+successfully!");
            exit();
        }
    }
}

// Get User Stats for Delete Modal (AJAX)
if (isset($_GET['get_user_stats'])) {
    header('Content-Type: application/json');
    $user_id = $_GET['user_id'] ?? 0;
    
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ? AND is_admin = 0");
    $stmt->execute([$user_id]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if ($user) {
        $stats = getUserStats($pdo, $user_id);
        echo json_encode([
            'success' => true,
            'user' => [
                'id' => $user['id'],
                'name' => $user['name'],
                'mobile' => $user['mobile'],
                'status' => $user['status'],
                'created_at' => $user['created_at']
            ],
            'stats' => $stats
        ]);
    } else {
        echo json_encode(['success' => false, 'message' => 'User not found']);
    }
    exit();
}

// Filters
$filter = $_GET['filter'] ?? 'all';
$search = $_GET['search'] ?? '';

$where = "WHERE is_admin = 0";
if ($filter == 'pending') $where .= " AND status = 'pending'";
elseif ($filter == 'active') $where .= " AND status = 'active'";
elseif ($filter == 'blocked') $where .= " AND status = 'blocked'";
elseif ($filter == 'rejected') $where .= " AND status = 'rejected'";

if ($search) {
    $where .= " AND (name LIKE :search OR mobile LIKE :search)";
}

$sql = "SELECT * FROM users $where ORDER BY created_at DESC";
$stmt = $pdo->prepare($sql);
if ($search) {
    $stmt->execute(['search' => "%$search%"]);
} else {
    $stmt->execute();
}
$users = $stmt->fetchAll();

// Get counts
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0");
$totalUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'active'");
$activeUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'blocked'");
$blockedUsers = $stmt->fetch()['c'];

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as total FROM wallet_requests WHERE status = 'pending'");
$pendingRequests = $stmt->fetch()['total'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Users - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        
        .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .stat-box { background: #1a1a1a; border-radius: 12px; padding: 20px; text-align: center; border: 1px solid #333; cursor: pointer; transition: all 0.3s; text-decoration: none; display: block; }
        .stat-box:hover { border-color: #ffd700; transform: translateY(-2px); }
        .stat-box.active { border-color: #ffd700; background: rgba(255,215,0,0.1); }
        .stat-number { font-size: 28px; font-weight: 700; color: #fff; }
        .stat-number.orange { color: #ff9800; }
        .stat-number.green { color: #00ff88; }
        .stat-number.red { color: #ff4444; }
        .stat-text { font-size: 12px; color: #888; margin-top: 5px; }
        
        .filter-bar { display: flex; gap: 15px; margin-bottom: 25px; flex-wrap: wrap; }
        .search-box { flex: 1; min-width: 250px; position: relative; }
        .search-input { width: 100%; padding: 14px 20px 14px 50px; background: #1a1a1a; border: 1px solid #333; border-radius: 10px; color: #fff; font-size: 14px; outline: none; }
        .search-input:focus { border-color: #ffd700; }
        .search-icon { position: absolute; left: 18px; top: 50%; transform: translateY(-50%); font-size: 18px; }
        .search-btn { padding: 14px 25px; background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%); border: none; border-radius: 10px; color: #000; font-weight: 700; cursor: pointer; }
        
        .users-table { width: 100%; background: #1a1a1a; border-radius: 15px; overflow: hidden; border: 1px solid #333; }
        .table-header { display: grid; grid-template-columns: 2fr 1.5fr 1fr 1fr 1.5fr; padding: 15px 20px; background: #252525; font-size: 12px; color: #888; text-transform: uppercase; }
        .table-row { display: grid; grid-template-columns: 2fr 1.5fr 1fr 1fr 1.5fr; padding: 15px 20px; border-bottom: 1px solid #333; align-items: center; }
        .table-row:hover { background: #252525; }
        
        .user-cell { display: flex; align-items: center; gap: 12px; }
        .user-avatar { width: 45px; height: 45px; border-radius: 50%; background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); display: flex; align-items: center; justify-content: center; font-weight: 700; font-size: 18px; color: #ffd700; }
        .user-info h4 { font-size: 14px; margin-bottom: 3px; }
        .user-info p { font-size: 12px; color: #888; }
        
        .status-badge { padding: 6px 12px; border-radius: 20px; font-size: 11px; font-weight: 700; text-transform: uppercase; }
        .status-badge.active { background: rgba(0,255,136,0.2); color: #00ff88; }
        .status-badge.pending { background: rgba(255,215,0,0.2); color: #ffd700; }
        .status-badge.blocked { background: rgba(255,68,68,0.2); color: #ff4444; }
        .status-badge.rejected { background: rgba(255,87,34,0.2); color: #ff5722; }
        
        .balance-cell { font-size: 15px; font-weight: 700; color: #00ff88; }
        
        .actions-cell { display: flex; gap: 8px; flex-wrap: wrap; }
        .btn-action { padding: 8px 12px; border: none; border-radius: 6px; font-size: 11px; font-weight: 600; cursor: pointer; transition: all 0.3s; }
        .btn-action.approve { background: #00c853; color: #fff; }
        .btn-action.reject { background: #ff5722; color: #fff; }
        .btn-action.block { background: #ff4444; color: #fff; }
        .btn-action.unblock { background: #2196f3; color: #fff; }
        .btn-action.delete { background: #333; color: #ff4444; }
        .btn-action.view { background: #333; color: #ffd700; }
        .btn-action:hover { opacity: 0.8; }
        
        .empty-state { text-align: center; padding: 60px 20px; color: #555; }
        .empty-state .icon { font-size: 50px; margin-bottom: 15px; }
        
        /* Modal Styles */
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.9); z-index: 1000; justify-content: center; align-items: center; padding: 20px; }
        .modal.active { display: flex; }
        .modal-content { background: #1a1a1a; border-radius: 20px; padding: 25px; max-width: 600px; width: 100%; max-height: 90vh; overflow-y: auto; border: 1px solid #333; }
        .modal-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 20px; padding-bottom: 15px; border-bottom: 1px solid #333; }
        .modal-title { font-size: 20px; font-weight: 700; color: #ffd700; }
        .modal-close { background: none; border: none; color: #888; font-size: 24px; cursor: pointer; }
        
        .detail-row { display: flex; justify-content: space-between; padding: 12px 0; border-bottom: 1px solid #333; }
        .detail-label { color: #888; font-size: 13px; }
        .detail-value { font-size: 14px; font-weight: 600; }
        
        /* Delete Confirmation Modal */
        .delete-modal-content { max-width: 550px; }
        .delete-warning { background: rgba(255,68,68,0.1); border: 1px solid #ff4444; border-radius: 12px; padding: 20px; margin-bottom: 20px; }
        .delete-warning-title { color: #ff4444; font-size: 18px; font-weight: 700; margin-bottom: 10px; display: flex; align-items: center; gap: 10px; }
        .delete-warning-text { font-size: 13px; color: #888; line-height: 1.6; }
        
        .user-summary { background: #252525; border-radius: 12px; padding: 20px; margin-bottom: 20px; }
        .user-summary-header { display: flex; align-items: center; gap: 15px; margin-bottom: 15px; padding-bottom: 15px; border-bottom: 1px solid #333; }
        .summary-avatar { width: 60px; height: 60px; background: linear-gradient(135deg, #8b0000, #5c0000); border-radius: 50%; display: flex; align-items: center; justify-content: center; font-size: 24px; font-weight: 800; color: #ffd700; }
        .summary-info h3 { font-size: 18px; margin-bottom: 5px; }
        .summary-info p { font-size: 13px; color: #888; }
        
        .stats-summary { display: grid; grid-template-columns: repeat(2, 1fr); gap: 12px; }
        .stat-item { background: #1a1a1a; padding: 15px; border-radius: 10px; }
        .stat-item-label { font-size: 11px; color: #888; margin-bottom: 5px; }
        .stat-item-value { font-size: 18px; font-weight: 700; }
        .stat-item-value.green { color: #00ff88; }
        .stat-item-value.red { color: #ff4444; }
        .stat-item-value.gold { color: #ffd700; }
        
        .archive-info { background: rgba(52,152,219,0.1); border: 1px solid #3498db; border-radius: 12px; padding: 15px; margin-bottom: 20px; }
        .archive-info-title { color: #3498db; font-weight: 700; margin-bottom: 8px; display: flex; align-items: center; gap: 8px; }
        .archive-info-text { font-size: 12px; color: #888; }
        
        .confirm-section { margin-top: 20px; }
        .confirm-label { font-size: 13px; color: #888; margin-bottom: 10px; }
        .confirm-input { width: 100%; padding: 15px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 16px; text-align: center; font-weight: 700; letter-spacing: 2px; outline: none; }
        .confirm-input:focus { border-color: #ff4444; }
        .confirm-hint { font-size: 11px; color: #ff4444; margin-top: 8px; text-align: center; }
        
        .modal-actions { display: flex; gap: 15px; margin-top: 20px; }
        .btn-modal { flex: 1; padding: 15px; border: none; border-radius: 10px; font-size: 14px; font-weight: 700; cursor: pointer; }
        .btn-modal.cancel { background: #333; color: #fff; }
        .btn-modal.delete { background: linear-gradient(135deg, #ff4444, #cc0000); color: #fff; }
        .btn-modal.delete:disabled { background: #333; color: #666; cursor: not-allowed; }
        
        .loading { text-align: center; padding: 40px; }
        .loading-spinner { width: 40px; height: 40px; border: 3px solid #333; border-top-color: #ffd700; border-radius: 50%; animation: spin 1s linear infinite; margin: 0 auto 15px; }
        @keyframes spin { to { transform: rotate(360deg); } }
        
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        
        @media (max-width: 900px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; padding-bottom: 80px; }
            .mobile-nav { display: block; }
            .table-header { display: none; }
            .table-row { grid-template-columns: 1fr; gap: 10px; }
            .stats-summary { grid-template-columns: 1fr; }
        }
        
        @media (max-width: 500px) {
            .stats-row { grid-template-columns: repeat(2, 1fr); }
            .filter-bar { flex-direction: column; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item active"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">👥 <span>Users Management</span></h1>
            </div>

            <?php if (isset($_GET['msg'])): ?><div class="alert success"><?= htmlspecialchars($_GET['msg']) ?></div><?php endif; ?>
            <?php if ($error): ?><div class="alert error"><?= $error ?></div><?php endif; ?>

            <div class="stats-row">
                <a href="users.php" class="stat-box <?= $filter == 'all' ? 'active' : '' ?>"><div class="stat-number"><?= $totalUsers ?></div><div class="stat-text">Total Users</div></a>
                <a href="users.php?filter=pending" class="stat-box <?= $filter == 'pending' ? 'active' : '' ?>"><div class="stat-number orange"><?= $pendingUsers ?></div><div class="stat-text">Pending</div></a>
                <a href="users.php?filter=active" class="stat-box <?= $filter == 'active' ? 'active' : '' ?>"><div class="stat-number green"><?= $activeUsers ?></div><div class="stat-text">Active</div></a>
                <a href="users.php?filter=blocked" class="stat-box <?= $filter == 'blocked' ? 'active' : '' ?>"><div class="stat-number red"><?= $blockedUsers ?></div><div class="stat-text">Blocked</div></a>
            </div>

            <form method="GET" class="filter-bar">
                <input type="hidden" name="filter" value="<?= htmlspecialchars($filter) ?>">
                <div class="search-box">
                    <span class="search-icon">🔍</span>
                    <input type="text" name="search" class="search-input" placeholder="Search name or mobile..." value="<?= htmlspecialchars($search) ?>">
                </div>
                <button type="submit" class="search-btn">Search</button>
            </form>

            <div class="users-table">
                <div class="table-header">
                    <div>User</div>
                    <div>Mobile</div>
                    <div>Balance</div>
                    <div>Status</div>
                    <div>Actions</div>
                </div>
                
                <?php if (empty($users)): ?>
                    <div class="empty-state"><div class="icon">👥</div><p>No users found</p></div>
                <?php else: ?>
                    <?php foreach ($users as $u): 
                        $userBalance = getWalletBalance($pdo, $u['id']);
                    ?>
                        <div class="table-row">
                            <div class="user-cell">
                                <div class="user-avatar"><?= strtoupper(substr($u['name'], 0, 1)) ?></div>
                                <div class="user-info">
                                    <h4><?= htmlspecialchars($u['name']) ?></h4>
                                    <p><?= date('d M Y', strtotime($u['created_at'])) ?></p>
                                </div>
                            </div>
                            <div><?= htmlspecialchars($u['mobile']) ?></div>
                            <div class="balance-cell">₹<?= number_format($userBalance, 0) ?></div>
                            <div><span class="status-badge <?= $u['status'] ?>"><?= $u['status'] ?></span></div>
                            <div class="actions-cell">
                                <?php if ($u['status'] == 'pending'): ?>
                                    <form method="POST" style="display:inline;" onsubmit="return promptCaptcha(this, 'approve')"><input type="hidden" name="user_id" value="<?= $u['id'] ?>"><input type="hidden" name="captcha_answer" class="captcha-input"><button type="submit" name="approve_user" class="btn-action approve">✓</button></form>
                                    <form method="POST" style="display:inline;" onsubmit="return promptCaptcha(this, 'reject')"><input type="hidden" name="user_id" value="<?= $u['id'] ?>"><input type="hidden" name="captcha_answer" class="captcha-input"><button type="submit" name="reject_user" class="btn-action reject">✕</button></form>
                                <?php elseif ($u['status'] == 'active'): ?>
                                    <form method="POST" style="display:inline;" onsubmit="return promptCaptcha(this, 'block')"><input type="hidden" name="user_id" value="<?= $u['id'] ?>"><input type="hidden" name="captcha_answer" class="captcha-input"><button type="submit" name="block_user" class="btn-action block">🚫</button></form>
                                <?php elseif ($u['status'] == 'blocked'): ?>
                                    <form method="POST" style="display:inline;" onsubmit="return promptCaptcha(this, 'unblock')"><input type="hidden" name="user_id" value="<?= $u['id'] ?>"><input type="hidden" name="captcha_answer" class="captcha-input"><button type="submit" name="unblock_user" class="btn-action unblock">✓</button></form>
                                <?php endif; ?>
                                <button class="btn-action view" onclick="viewUser(<?= htmlspecialchars(json_encode($u)) ?>, <?= $userBalance ?>)">👁</button>
                                <button class="btn-action" style="background:#9b59b6;" onclick="openPasswordModal(<?= $u['id'] ?>, '<?= htmlspecialchars($u['name']) ?>', '<?= $u['mobile'] ?>')" title="Reset Password">🔑</button>
                                <button class="btn-action delete" onclick="openDeleteModal(<?= $u['id'] ?>)">🗑</button>
                            </div>
                        </div>
                    <?php endforeach; ?>
                <?php endif; ?>
            </div>
        </main>
    </div>

    <!-- View User Modal -->
    <div class="modal" id="userModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">👤 User Details</div>
                <button class="modal-close" onclick="closeModal('userModal')">&times;</button>
            </div>
            <div class="modal-body" id="userModalBody"></div>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content delete-modal-content">
            <div class="modal-header">
                <div class="modal-title">🗑️ Delete User</div>
                <button class="modal-close" onclick="closeModal('deleteModal')">&times;</button>
            </div>
            
    <!-- Password Reset Modal -->
    <div class="modal" id="passwordModal">
        <div class="modal-content" style="max-width:400px;">
            <div class="modal-header">
                <div class="modal-title">🔑 Reset Password</div>
                <button class="modal-close" onclick="closeModal('passwordModal')">&times;</button>
            </div>
            <div class="modal-body">
                <form method="POST" id="passwordForm">
                    <input type="hidden" name="user_id" id="pwdUserId">
                    <div style="text-align:center;margin-bottom:20px;">
                        <div style="font-size:40px;margin-bottom:10px;">👤</div>
                        <div style="font-size:16px;font-weight:bold;color:#ffd700;" id="pwdUserName"></div>
                        <div style="font-size:14px;color:#888;" id="pwdUserMobile"></div>
                    </div>
                    <div class="form-group">
                        <label class="form-label">New Password</label>
                        <input type="text" name="new_password" class="form-input" placeholder="Enter new password (min 6 chars)" required minlength="6">
                        <small style="color:#888;display:block;margin-top:5px;">User will login with this password</small>
                    </div>
                    <div class="form-group">
                        <label class="form-label" style="color:#ffd700;">🔢 Captcha: <?= generateAdminCaptcha('password') ?></label>
                        <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                    </div>
                    <div class="modal-actions">
                        <button type="button" class="btn-modal cancel" onclick="closeModal('passwordModal')">Cancel</button>
                        <button type="submit" name="reset_password" class="btn-modal" style="background:linear-gradient(135deg, #9b59b6, #8e44ad);">🔑 Reset Password</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
            <div id="deleteModalBody">
                <div class="loading">
                    <div class="loading-spinner"></div>
                    <p>Loading user data...</p>
                </div>
            </div>
        </div>
    </div>

    <!-- Mobile Nav -->
    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item active"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="results.php" class="mobile-nav-item"><span class="mobile-nav-icon">🏆</span><span class="mobile-nav-text">Results</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>

    <script>
        function viewUser(user, balance) {
            let qrCodeHtml = '';
            if (user.qr_code) {
                qrCodeHtml = `
                    <div class="detail-row" style="flex-direction:column;align-items:flex-start;">
                        <span class="detail-label" style="margin-bottom:10px;">Payment QR Code</span>
                        <img src="../uploads/${user.qr_code}" alt="User QR" style="max-width:200px;border-radius:8px;border:2px solid #333;">
                    </div>
                `;
            }
            
            let profilePhotoHtml = '';
            if (user.profile_photo) {
                profilePhotoHtml = `
                    <div class="detail-row" style="flex-direction:column;align-items:flex-start;">
                        <span class="detail-label" style="margin-bottom:10px;">Profile Photo</span>
                        <img src="../uploads/${user.profile_photo}" alt="Profile" style="max-width:100px;border-radius:50%;border:2px solid #ffd700;">
                    </div>
                `;
            }
            
            document.getElementById('userModalBody').innerHTML = `
                ${profilePhotoHtml}
                <div class="detail-row"><span class="detail-label">Name</span><span class="detail-value">${user.name}</span></div>
                <div class="detail-row"><span class="detail-label">Mobile</span><span class="detail-value">${user.mobile}</span></div>
                <div class="detail-row"><span class="detail-label">Balance</span><span class="detail-value" style="color:#00ff88">₹${balance.toLocaleString()}</span></div>
                <div class="detail-row"><span class="detail-label">Status</span><span class="detail-value">${user.status.toUpperCase()}</span></div>
                <div class="detail-row"><span class="detail-label">UPI</span><span class="detail-value">${user.upi_id || 'Not set'}</span></div>
                <div class="detail-row"><span class="detail-label">Bank</span><span class="detail-value">${user.bank_name || 'Not set'}</span></div>
                <div class="detail-row"><span class="detail-label">Account</span><span class="detail-value">${user.account_no || 'Not set'}</span></div>
                <div class="detail-row"><span class="detail-label">Account Holder</span><span class="detail-value">${user.account_holder || 'Not set'}</span></div>
                <div class="detail-row"><span class="detail-label">IFSC</span><span class="detail-value">${user.ifsc_code || 'Not set'}</span></div>
                <div class="detail-row"><span class="detail-label">Joined</span><span class="detail-value">${new Date(user.created_at).toLocaleDateString('en-IN')}</span></div>
                ${qrCodeHtml}
            `;
            document.getElementById('userModal').classList.add('active');
        }
        
        function openDeleteModal(userId) {
            document.getElementById('deleteModal').classList.add('active');
            document.getElementById('deleteModalBody').innerHTML = `
                <div class="loading">
                    <div class="loading-spinner"></div>
                    <p>Loading user data & calculating stats...</p>
                </div>
            `;
            
            // Fetch user stats
            fetch('users.php?get_user_stats=1&user_id=' + userId)
                .then(res => res.json())
                .then(data => {
                    if (data.success) {
                        renderDeleteConfirmation(data.user, data.stats);
                    } else {
                        document.getElementById('deleteModalBody').innerHTML = `
                            <div class="delete-warning">
                                <div class="delete-warning-title">❌ Error</div>
                                <div class="delete-warning-text">${data.message}</div>
                            </div>
                            <button class="btn-modal cancel" onclick="closeModal('deleteModal')" style="width:100%">Close</button>
                        `;
                    }
                })
                .catch(err => {
                    document.getElementById('deleteModalBody').innerHTML = `
                        <div class="delete-warning">
                            <div class="delete-warning-title">❌ Error</div>
                            <div class="delete-warning-text">Failed to load user data. Please try again.</div>
                        </div>
                        <button class="btn-modal cancel" onclick="closeModal('deleteModal')" style="width:100%">Close</button>
                    `;
                });
        }
        
        function openPasswordModal(userId, userName, userMobile) {
            document.getElementById('pwdUserId').value = userId;
            document.getElementById('pwdUserName').textContent = userName;
            document.getElementById('pwdUserMobile').textContent = userMobile;
            document.getElementById('passwordModal').classList.add('active');
        }
        
        function renderDeleteConfirmation(user, stats) {
            const hasPending = stats.pending_bids > 0 || stats.pending_requests > 0;
            
            let warningHtml = '';
            if (hasPending) {
                warningHtml = `
                    <div class="delete-warning">
                        <div class="delete-warning-title">⚠️ Cannot Delete User</div>
                        <div class="delete-warning-text">
                            ${stats.pending_bids > 0 ? `<p>• User has <strong>${stats.pending_bids} pending bids</strong>. Please publish results first.</p>` : ''}
                            ${stats.pending_requests > 0 ? `<p>• User has <strong>${stats.pending_requests} pending wallet requests</strong>. Please approve/reject them first.</p>` : ''}
                        </div>
                    </div>
                `;
            } else {
                warningHtml = `
                    <div class="delete-warning">
                        <div class="delete-warning-title">⚠️ Permanent Action</div>
                        <div class="delete-warning-text">
                            This will permanently delete the user from the main database. All data will be archived for record keeping but cannot be restored to active status.
                        </div>
                    </div>
                `;
            }
            
            document.getElementById('deleteModalBody').innerHTML = `
                ${warningHtml}
                
                <div class="user-summary">
                    <div class="user-summary-header">
                        <div class="summary-avatar">${user.name.charAt(0).toUpperCase()}</div>
                        <div class="summary-info">
                            <h3>${user.name}</h3>
                            <p>📱 ${user.mobile} • Joined ${new Date(user.created_at).toLocaleDateString('en-IN')}</p>
                        </div>
                    </div>
                    
                    <div class="stats-summary">
                        <div class="stat-item">
                            <div class="stat-item-label">Current Balance</div>
                            <div class="stat-item-value ${stats.balance >= 0 ? 'green' : 'red'}">₹${parseFloat(stats.balance).toLocaleString()}</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-item-label">Total Bids</div>
                            <div class="stat-item-value">${stats.total_bids}</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-item-label">Total Bid Amount</div>
                            <div class="stat-item-value gold">₹${parseFloat(stats.total_bid_amount).toLocaleString()}</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-item-label">Wins / Win Amount</div>
                            <div class="stat-item-value green">${stats.total_wins} / ₹${parseFloat(stats.total_win_amount).toLocaleString()}</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-item-label">Total Deposits</div>
                            <div class="stat-item-value green">₹${parseFloat(stats.total_deposits).toLocaleString()}</div>
                        </div>
                        <div class="stat-item">
                            <div class="stat-item-label">Total Withdrawals</div>
                            <div class="stat-item-value red">₹${parseFloat(stats.total_withdrawals).toLocaleString()}</div>
                        </div>
                    </div>
                </div>
                
                <div class="archive-info">
                    <div class="archive-info-title">📦 Data to be Archived</div>
                    <div class="archive-info-text">
                        • ${stats.total_bids} bids (₹${parseFloat(stats.total_bid_amount).toLocaleString()} total)<br>
                        • ${stats.wallet_txns} wallet transactions<br>
                        • ${stats.wallet_requests} wallet requests<br>
                        • User profile & payment details
                    </div>
                </div>
                
                ${!hasPending ? `
                    <form method="POST" id="deleteForm">
                        <input type="hidden" name="user_id" value="${user.id}">
                        <div class="confirm-section">
                            <div class="confirm-label">To confirm deletion, type the user's mobile number:</div>
                            <input type="text" name="confirm_text" id="confirmInput" class="confirm-input" placeholder="${user.mobile}" autocomplete="off">
                            <div class="confirm-hint">Type: ${user.mobile}</div>
                        </div>
                        <div class="form-group" style="margin:15px 0;">
                            <label class="form-label" style="color:#ffd700;">🔢 Captcha: <?= generateAdminCaptcha('delete_user') ?></label>
                            <input type="number" name="captcha_answer" class="confirm-input" placeholder="Enter answer" required style="margin-top:5px;">
                        </div>
                        <div class="modal-actions">
                            <button type="button" class="btn-modal cancel" onclick="closeModal('deleteModal')">Cancel</button>
                            <button type="submit" name="confirm_delete_user" class="btn-modal delete" id="deleteBtn" disabled>🗑️ Delete & Archive</button>
                        </div>
                    </form>
                ` : `
                    <div class="modal-actions">
                        <button type="button" class="btn-modal cancel" onclick="closeModal('deleteModal')" style="width:100%">Close</button>
                    </div>
                `}
            `;
            
            // Enable delete button only when mobile matches
            if (!hasPending) {
                const confirmInput = document.getElementById('confirmInput');
                const deleteBtn = document.getElementById('deleteBtn');
                
                confirmInput.addEventListener('input', function() {
                    deleteBtn.disabled = this.value.toUpperCase() !== user.mobile.toUpperCase();
                });
            }
        }
        
        function closeModal(id) {
            document.getElementById(id).classList.remove('active');
        }
        
        // Client-side Captcha for inline action buttons
        function promptCaptcha(form, action) {
            var num1 = Math.floor(Math.random() * 20) + 1;
            var num2 = Math.floor(Math.random() * 20) + 1;
            var correctAnswer = num1 + num2;
            var answer = prompt('🔢 Solve to ' + action + ':\n\n' + num1 + ' + ' + num2 + ' = ?');
            
            if (answer === null) return false;
            
            if (parseInt(answer) !== correctAnswer) {
                alert('❌ Wrong answer! Action cancelled.');
                return false;
            }
            
            // Pass the answer for form submission
            form.querySelector('.captcha-input').value = answer;
            return true;
        }
        
        // Close on outside click
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', function(e) {
                if (e.target === this) {
                    this.classList.remove('active');
                }
            });
        });
    </script>
</body>
</html>