<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

// Initialize secure session
initSecureSession();

if (isLoggedIn()) {
    header("Location: index.php");
    exit();
}

$error = '';
$success = '';
$showApprovalMsg = false;
$referralApplied = false;

// Get settings
$auto_approve = getSetting($pdo, 'auto_approve', '1');
$whatsapp_number = getSetting($pdo, 'whatsapp_number', '');
$telegram_link = getSetting($pdo, 'telegram_link', '');
$referrerBonus = getSetting($pdo, 'referral_bonus_referrer', '50');
$referredBonus = getSetting($pdo, 'referral_bonus_referred', '25');

// Check for referral code in URL
$refCode = $_GET['ref'] ?? '';

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    // Verify CSRF
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh and try again.';
    } else {
        $name = trim($_POST['name'] ?? '');
        $mobile = trim($_POST['mobile'] ?? '');
        $password = $_POST['password'] ?? '';
        $captcha = $_POST['captcha'] ?? '';
        $referralCode = trim($_POST['referral_code'] ?? '');
        
        if (empty($name) || empty($mobile) || empty($password) || empty($captcha)) {
            $error = 'All fields are required';
        } elseif (!verifyUserCaptcha($captcha)) {
            $error = 'Wrong captcha answer';
        } elseif (!preg_match('/^[0-9]{10}$/', $mobile)) {
            $error = 'Enter valid 10 digit mobile number';
        } elseif (strlen($password) < 6) {
            $error = 'Password must be at least 6 characters';
        } else {
            $stmt = $pdo->prepare("SELECT id FROM users WHERE mobile = ?");
            $stmt->execute([$mobile]);
            
            if ($stmt->fetch()) {
                $error = 'Mobile number already registered';
            } else {
                $hash = password_hash($password, PASSWORD_DEFAULT);
                $status = ($auto_approve == '1') ? 'active' : 'pending';
                
                // Generate unique referral code for new user
                $newUserRefCode = generateReferralCode(time());
                
                $pdo->beginTransaction();
                try {
                    $stmt = $pdo->prepare("INSERT INTO users (name, mobile, password, status, referral_code) VALUES (?, ?, ?, ?, ?)");
                    $stmt->execute([$name, $mobile, $hash, $status, $newUserRefCode]);
                    $newUserId = $pdo->lastInsertId();
                    
                    // Update referral code with user ID for uniqueness
                    $finalRefCode = generateReferralCode($newUserId);
                    $stmt = $pdo->prepare("UPDATE users SET referral_code = ? WHERE id = ?");
                    $stmt->execute([$finalRefCode, $newUserId]);
                    
                    // Apply referral if provided
                    if (!empty($referralCode)) {
                        $refResult = applyReferralCode($pdo, $newUserId, $referralCode);
                        if ($refResult['success']) {
                            $referralApplied = true;
                        }
                    }
                    
                    // Add welcome notification
                    addNotification($pdo, $newUserId, 'welcome', 'Welcome to Satta King! 🎰', 
                        'Your account has been created. ' . ($status == 'pending' ? 'Please wait for admin approval.' : 'You can start playing now!'));
                    
                    $pdo->commit();
                    clearUserCaptcha();
                    
                    // Log security event
                    logSecurityEvent($newUserId, 'USER_REGISTERED', 'low', "New user registered: $name ($mobile)");
                    
                    if ($auto_approve == '1') {
                        $success = 'Registration successful! Please login.';
                        if ($referralApplied) {
                            $success .= " You'll get ₹$referredBonus bonus on first deposit!";
                        }
                    } else {
                        $showApprovalMsg = true;
                    }
                } catch (Exception $e) {
                    $pdo->rollBack();
                    $error = 'Something went wrong. Please try again.';
                }
            }
        }
    }
}

// Generate captcha for display
$captchaText = generateUserCaptcha();
$csrfToken = generateCSRFToken();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Register - Satta King</title>
    <link rel="stylesheet" href="assets/css/style.css">
</head>
<body>
    <div class="auth-container">
        <div class="auth-box">
            <h1>🎰 SATTA KING</h1>
            
            <?php if ($error): ?>
                <div class="alert alert-error"><?= htmlspecialchars($error) ?></div>
            <?php endif; ?>
            
            <?php if ($success): ?>
                <div class="alert alert-success"><?= htmlspecialchars($success) ?></div>
            <?php endif; ?>
            
            <?php if ($showApprovalMsg): ?>
                <div class="alert alert-success">
                    Registration successful!<br>
                    Your account is pending approval.
                    <?php if ($referralApplied): ?>
                        <br>🎁 Referral applied! You'll get ₹<?= $referredBonus ?> on first deposit.
                    <?php endif; ?>
                </div>
                <div class="approval-buttons">
                    <?php if ($whatsapp_number): ?>
                        <a href="https://wa.me/<?= htmlspecialchars($whatsapp_number) ?>?text=<?= urlencode('Hi, I have registered on Satta King with mobile: ' . $mobile . '. Please approve my account.') ?>" class="btn btn-whatsapp" target="_blank">
                            📱 Request on WhatsApp
                        </a>
                    <?php endif; ?>
                    <?php if ($telegram_link): ?>
                        <a href="<?= htmlspecialchars($telegram_link) ?>" class="btn btn-telegram" target="_blank">
                            ✈️ Request on Telegram
                        </a>
                    <?php endif; ?>
                </div>
                <p class="auth-link"><a href="login.php">Back to Login</a></p>
            <?php else: ?>
                <form method="POST">
                    <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                    
                    <div class="form-group">
                        <label>Full Name</label>
                        <input type="text" name="name" placeholder="Enter your name" value="<?= htmlspecialchars($_POST['name'] ?? '') ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Mobile Number</label>
                        <input type="tel" name="mobile" placeholder="10 digit mobile number" maxlength="10" value="<?= htmlspecialchars($_POST['mobile'] ?? '') ?>" required>
                    </div>
                    
                    <div class="form-group">
                        <label>Password</label>
                        <input type="password" name="password" placeholder="Create password (min 6 chars)" required>
                    </div>
                    
                    <div class="form-group">
                        <label>🎁 Referral Code (Optional)</label>
                        <input type="text" name="referral_code" placeholder="Enter referral code" value="<?= htmlspecialchars($refCode ?: ($_POST['referral_code'] ?? '')) ?>" style="text-transform:uppercase;">
                        <small style="color:#888;font-size:11px;">Get ₹<?= $referredBonus ?> bonus on first deposit!</small>
                    </div>
                    
                    <div class="form-group">
                        <label>🔢 Captcha: <strong style="color:#ffd700;"><?= $captchaText ?></strong></label>
                        <input type="number" name="captcha" placeholder="Enter answer" required>
                    </div>
                    
                    <button type="submit" class="btn">REGISTER</button>
                </form>
                
                <p class="auth-link">Already have account? <a href="login.php">Login</a></p>
            <?php endif; ?>
        </div>
    </div>
</body>
</html>
