<?php
require_once 'config/db.php';
require_once 'config/auth.php';
require_once 'config/functions.php';
require_once 'config/security.php';

// Initialize secure session
initSecureSession();
requireLogin();

// Check session timeout
if (!checkSessionTimeout($pdo)) {
    header("Location: login.php?msg=Session+expired");
    exit();
}

$user = getUser($pdo);
if (!$user) {
    session_destroy();
    header("Location: login.php");
    exit();
}

// Track session
trackUserSession($pdo, $user['id']);

$balance = getWalletBalance($pdo, $user['id']);

$game_id = $_GET['id'] ?? 0;
$stmt = $pdo->prepare("SELECT * FROM games WHERE id = ? AND status = 'active'");
$stmt->execute([$game_id]);
$game = $stmt->fetch(PDO::FETCH_ASSOC);

if (!$game) {
    header("Location: index.php");
    exit();
}

$isOpen = isGameOpen($pdo, $game);
$playingDate = getGamePlayingDate($game);

if (!$isOpen) {
    header("Location: index.php?msg=closed");
    exit();
}

$error = '';
$success = '';

// Handle Bid Cancellation
if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['cancel_bid'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
        $bidId = intval($_POST['bid_id']);
        $result = cancelBid($pdo, $bidId, $user['id']);
        if ($result['success']) {
            $success = $result['message'];
            $balance = getWalletBalance($pdo, $user['id']);
        } else {
            $error = $result['message'];
        }
    }
}

if ($_SERVER['REQUEST_METHOD'] == 'POST' && isset($_POST['place_bid'])) {
    if (!verifyCSRFToken($_POST['csrf_token'] ?? '')) {
        $error = 'Invalid request. Please refresh.';
    } else {
        $bids = json_decode($_POST['bids_data'], true);
        $captcha_answer = $_POST['captcha_answer'] ?? '';
        $captcha_expected = $_SESSION['captcha_answer'] ?? '';
        
        if ($captcha_answer != $captcha_expected) {
            $error = 'Wrong captcha answer!';
        } elseif (empty($bids)) {
            $error = 'Please add at least one bid';
        } else {
            $totalAmount = 0;
            foreach ($bids as $bid) {
                $totalAmount += floatval($bid['amount']);
            }
            
            // Check daily bid limit
            $limitCheck = checkDailyBidLimit($pdo, $user['id'], $totalAmount);
            if (!$limitCheck['allowed']) {
                $error = $limitCheck['message'];
            } elseif ($totalAmount > $balance) {
                $error = 'Insufficient balance! Need ₹' . number_format($totalAmount, 2);
            } else {
                // Rate limiting - max 50 bids per minute
                $rateCheck = checkRateLimit($pdo, $user['id'], 'bid', 50, 60);
                if (!$rateCheck['allowed']) {
                    $error = 'Too many bids! Please wait a moment.';
                } else {
                    $pdo->beginTransaction();
                    try {
                        foreach ($bids as $bid) {
                            $number = $bid['number'];
                            if ($bid['type'] == 'jodi') {
                                $number = str_pad($number, 2, '0', STR_PAD_LEFT);
                            }
                            $amount = floatval($bid['amount']);
                            $bid_type = $bid['type'];
                            
                            // Insert bid with archive columns
                            $stmt = $pdo->prepare("INSERT INTO bids (user_id, user_name_archive, user_mobile_archive, game_id, game_name_archive, playing_date, bid_type, number, amount) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?)");
                            $stmt->execute([$user['id'], $user['name'], $user['mobile'], $game['id'], $game['name'], $playingDate, $bid_type, $number, $amount]);
                            
                            // Insert wallet debit with archive columns
                            $stmt = $pdo->prepare("INSERT INTO wallet (user_id, user_name_archive, user_mobile_archive, type, amount, description) VALUES (?, ?, ?, 'debit', ?, ?)");
                            $desc = $game['name'] . " - " . ucfirst($bid_type) . " - " . $number;
                            $stmt->execute([$user['id'], $user['name'], $user['mobile'], $amount, $desc]);
                        }
                        
                        // Update daily bid total
                        updateDailyBidTotal($pdo, $user['id'], $totalAmount);
                        
                        // Add notification
                        addNotification($pdo, $user['id'], 'bid_placed', 'Bid Placed Successfully', 
                            "Your bid of ₹" . number_format($totalAmount, 0) . " on {$game['name']} has been placed.");
                        
                        $pdo->commit();
                        $success = 'Bids placed successfully! Total: ₹' . number_format($totalAmount, 2);
                        $balance = getWalletBalance($pdo, $user['id']);
                    } catch (Exception $e) {
                        $pdo->rollBack();
                        $error = 'Something went wrong!';
                    }
                }
            }
        }
    }
}

$num1 = rand(1, 9);
$num2 = rand(1, 9);
$_SESSION['captcha_answer'] = $num1 + $num2;

$min_bid = getSetting($pdo, 'min_bid', '10');
$max_bid = getSetting($pdo, 'max_bid', '10000');
$csrfToken = generateCSRFToken();

// Get daily bid limit info
$dailyBidLimit = getSetting($pdo, 'daily_bid_limit', 100000);
$bidCancelMinutes = getSetting($pdo, 'bid_cancel_minutes', 5);

// Get user's today bids for this game (for cancellation)
$stmt = $pdo->prepare("SELECT b.*, g.close_time, g.is_overnight FROM bids b 
    JOIN games g ON b.game_id = g.id 
    WHERE b.user_id = ? AND b.game_id = ? AND b.playing_date = ? AND b.status = 'pending' AND b.is_cancelled = 0
    ORDER BY b.created_at DESC");
$stmt->execute([$user['id'], $game['id'], $playingDate]);
$todayBids = $stmt->fetchAll(PDO::FETCH_ASSOC);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Play <?= htmlspecialchars($game['name']) ?> - Satta King</title>
    <style>
        * {
            margin: 0;
            padding: 0;
            box-sizing: border-box;
        }
        
        body {
            font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif;
            background: #0f0f0f;
            color: #fff;
            min-height: 100vh;
            padding-bottom: 30px;
        }
        
        /* Header */
        .header {
            background: linear-gradient(135deg, #8b0000 0%, #1a1a1a 100%);
            padding: 15px 20px;
            display: flex;
            justify-content: space-between;
            align-items: center;
            position: sticky;
            top: 0;
            z-index: 100;
        }
        
        .back-btn {
            color: #fff;
            text-decoration: none;
            font-size: 16px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        .game-title {
            font-size: 18px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .wallet-display {
            background: rgba(255,215,0,0.2);
            padding: 8px 15px;
            border-radius: 20px;
            font-size: 14px;
            font-weight: 600;
            color: #ffd700;
        }
        
        /* Container */
        .container {
            padding: 20px;
            max-width: 500px;
            margin: 0 auto;
        }
        
        /* Alert */
        .alert {
            padding: 15px;
            border-radius: 10px;
            margin-bottom: 20px;
            font-size: 14px;
        }
        
        .alert-error {
            background: rgba(255,0,0,0.2);
            border: 1px solid #ff4444;
            color: #ff6666;
        }
        
        .alert-success {
            background: rgba(0,255,0,0.2);
            border: 1px solid #00cc00;
            color: #00ff00;
        }
        
        /* Game Info Card */
        .game-info {
            background: linear-gradient(135deg, #1a1a2e 0%, #16213e 100%);
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        
        .info-grid {
            display: grid;
            grid-template-columns: repeat(2, 1fr);
            gap: 15px;
        }
        
        .info-item {
            text-align: center;
            padding: 12px;
            background: rgba(0,0,0,0.3);
            border-radius: 10px;
        }
        
        .info-label {
            font-size: 11px;
            color: #888;
            text-transform: uppercase;
        }
        
        .info-value {
            font-size: 16px;
            font-weight: 700;
            color: #fff;
            margin-top: 5px;
        }
        
        .info-value.gold {
            color: #ffd700;
        }
        
        .info-value.green {
            color: #00ff88;
        }
        
        /* Bid Type Tabs */
        .bid-tabs {
            display: flex;
            gap: 10px;
            margin-bottom: 20px;
        }
        
        .bid-tab {
            flex: 1;
            padding: 15px 10px;
            background: #1a1a1a;
            border: 2px solid #333;
            border-radius: 12px;
            text-align: center;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .bid-tab:hover {
            border-color: #555;
        }
        
        .bid-tab.active {
            border-color: #ffd700;
            background: rgba(255,215,0,0.1);
        }
        
        .tab-icon {
            font-size: 24px;
            margin-bottom: 5px;
        }
        
        .tab-name {
            font-size: 12px;
            font-weight: 700;
            color: #fff;
        }
        
        .tab-rate {
            font-size: 11px;
            color: #ffd700;
            margin-top: 3px;
        }
        
        /* Bid Input Section */
        .bid-section {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        
        .section-title {
            color: #ffd700;
            font-size: 16px;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            gap: 8px;
        }
        
        /* Jodi Input */
        .jodi-row {
            display: flex;
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .jodi-number {
            width: 100px;
            padding: 15px;
            background: #252525;
            border: 2px solid #444;
            border-radius: 10px;
            color: #ffd700;
            font-size: 24px;
            font-weight: 700;
            text-align: center;
            outline: none;
        }
        
        .jodi-number:focus {
            border-color: #ffd700;
        }
        
        .amount-input {
            flex: 1;
            padding: 15px;
            background: #252525;
            border: 2px solid #444;
            border-radius: 10px;
            color: #fff;
            font-size: 16px;
            outline: none;
        }
        
        .amount-input:focus {
            border-color: #ffd700;
        }
        
        .btn-add {
            padding: 15px 25px;
            background: linear-gradient(135deg, #ffd700 0%, #ff8c00 100%);
            border: none;
            border-radius: 10px;
            color: #000;
            font-size: 14px;
            font-weight: 700;
            cursor: pointer;
        }
        
        /* Digit Grid for Andar/Bahar */
        .digit-grid {
            display: grid;
            grid-template-columns: repeat(5, 1fr);
            gap: 10px;
            margin-bottom: 15px;
        }
        
        .digit-btn {
            aspect-ratio: 1;
            background: #252525;
            border: 2px solid #444;
            border-radius: 12px;
            color: #fff;
            font-size: 22px;
            font-weight: 700;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .digit-btn:hover {
            border-color: #ffd700;
            background: rgba(255,215,0,0.1);
        }
        
        .digit-btn.selected {
            background: #ffd700;
            color: #000;
            border-color: #ffd700;
        }
        
        .amount-row {
            display: flex;
            gap: 10px;
        }
        
        .full-input {
            flex: 1;
            padding: 15px;
            background: #252525;
            border: 2px solid #444;
            border-radius: 10px;
            color: #fff;
            font-size: 16px;
            outline: none;
        }
        
        .full-input:focus {
            border-color: #ffd700;
        }
        
        /* Hint Box */
        .hint-box {
            background: rgba(255,215,0,0.1);
            border: 1px solid rgba(255,215,0,0.3);
            border-radius: 10px;
            padding: 12px 15px;
            margin-top: 15px;
            font-size: 12px;
            color: #ffd700;
        }
        
        /* Min Max Info */
        .min-max {
            text-align: center;
            font-size: 12px;
            color: #888;
            margin-top: 10px;
        }
        
        /* Bids List */
        .bids-container {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
        }
        
        .bids-header {
            display: flex;
            justify-content: space-between;
            align-items: center;
            margin-bottom: 15px;
        }
        
        .bids-title {
            color: #ffd700;
            font-size: 16px;
        }
        
        .bids-count {
            background: #ffd700;
            color: #000;
            padding: 3px 10px;
            border-radius: 15px;
            font-size: 12px;
            font-weight: 700;
        }
        
        .bids-list {
            max-height: 300px;
            overflow-y: auto;
        }
        
        .bid-item {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 12px 15px;
            background: #252525;
            border-radius: 10px;
            margin-bottom: 10px;
        }
        
        .bid-left {
            display: flex;
            align-items: center;
            gap: 12px;
        }
        
        .bid-type {
            padding: 5px 10px;
            border-radius: 6px;
            font-size: 10px;
            font-weight: 700;
            text-transform: uppercase;
        }
        
        .bid-type.jodi {
            background: #3498db;
        }
        
        .bid-type.andar {
            background: #2ecc71;
        }
        
        .bid-type.bahar {
            background: #e74c3c;
        }
        
        .bid-number {
            font-size: 20px;
            font-weight: 700;
            color: #ffd700;
        }
        
        .bid-right {
            text-align: right;
        }
        
        .bid-amount {
            font-size: 14px;
            font-weight: 600;
            color: #fff;
        }
        
        .bid-win {
            font-size: 11px;
            color: #00ff88;
        }
        
        .bid-remove {
            background: rgba(255,0,0,0.2);
            border: none;
            color: #ff4444;
            width: 28px;
            height: 28px;
            border-radius: 50%;
            font-size: 16px;
            cursor: pointer;
            margin-left: 10px;
        }
        
        .bids-empty {
            text-align: center;
            padding: 30px;
            color: #555;
        }
        
        .bids-total {
            display: flex;
            justify-content: space-between;
            align-items: center;
            padding: 15px;
            background: #252525;
            border-radius: 10px;
            margin-top: 15px;
        }
        
        .total-label {
            font-size: 14px;
            color: #888;
        }
        
        .total-amount {
            font-size: 20px;
            font-weight: 700;
            color: #ffd700;
        }
        
        /* Captcha */
        .captcha-box {
            background: #1a1a1a;
            border-radius: 15px;
            padding: 20px;
            margin-bottom: 20px;
            border: 1px solid #333;
            text-align: center;
        }
        
        .captcha-question {
            font-size: 24px;
            color: #ffd700;
            margin-bottom: 15px;
        }
        
        .captcha-input {
            width: 120px;
            padding: 15px;
            background: #252525;
            border: 2px solid #444;
            border-radius: 10px;
            color: #fff;
            font-size: 20px;
            font-weight: 700;
            text-align: center;
            outline: none;
        }
        
        .captcha-input:focus {
            border-color: #ffd700;
        }
        
        /* Submit Button */
        .btn-submit {
            width: 100%;
            padding: 18px;
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            border: none;
            border-radius: 12px;
            color: #fff;
            font-size: 16px;
            font-weight: 700;
            cursor: pointer;
            text-transform: uppercase;
            letter-spacing: 1px;
        }
        
        .btn-submit:disabled {
            background: #333;
            color: #666;
            cursor: not-allowed;
        }
        
        /* Modal */
        .modal {
            display: none;
            position: fixed;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background: rgba(0,0,0,0.9);
            z-index: 1000;
            justify-content: center;
            align-items: center;
            padding: 20px;
        }
        
        .modal.active {
            display: flex;
        }
        
        .modal-content {
            background: #1a1a1a;
            border-radius: 20px;
            padding: 25px;
            max-width: 400px;
            width: 100%;
            max-height: 80vh;
            overflow-y: auto;
            border: 1px solid #333;
        }
        
        .modal-title {
            color: #ffd700;
            font-size: 20px;
            margin-bottom: 20px;
            text-align: center;
        }
        
        .modal-info {
            background: #252525;
            border-radius: 10px;
            padding: 15px;
            margin-bottom: 20px;
        }
        
        .modal-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #333;
        }
        
        .modal-row:last-child {
            border-bottom: none;
        }
        
        .modal-label {
            color: #888;
        }
        
        .modal-value {
            font-weight: 600;
        }
        
        .modal-value.green {
            color: #00ff88;
        }
        
        .modal-value.gold {
            color: #ffd700;
        }
        
        .modal-bids {
            max-height: 200px;
            overflow-y: auto;
            margin-bottom: 20px;
        }
        
        .modal-bid {
            display: flex;
            justify-content: space-between;
            padding: 10px;
            background: #252525;
            border-radius: 8px;
            margin-bottom: 8px;
            font-size: 13px;
        }
        
        .modal-actions {
            display: flex;
            gap: 15px;
        }
        
        .modal-btn {
            flex: 1;
            padding: 15px;
            border: none;
            border-radius: 10px;
            font-size: 14px;
            font-weight: 700;
            cursor: pointer;
        }
        
        .modal-btn.cancel {
            background: #333;
            color: #fff;
        }
        
        .modal-btn.confirm {
            background: linear-gradient(135deg, #00c853 0%, #00a844 100%);
            color: #fff;
        }
    </style>
</head>
<body>
    <header class="header">
        <a href="index.php" class="back-btn">← Back</a>
        <div class="game-title"><?= htmlspecialchars($game['name']) ?></div>
        <div class="wallet-display">₹<?= number_format($balance, 2) ?></div>
    </header>

    <div class="container">
        <?php if ($error): ?>
            <div class="alert alert-error"><?= $error ?></div>
        <?php endif; ?>
        <?php if ($success): ?>
            <div class="alert alert-success"><?= $success ?></div>
        <?php endif; ?>

        <!-- Game Info -->
        <div class="game-info">
            <div class="info-grid">
                <div class="info-item">
                    <div class="info-label">Close Time</div>
                    <div class="info-value"><?= date('h:i A', strtotime($game['close_time'])) ?></div>
                </div>
                <div class="info-item">
                    <div class="info-label">Rate</div>
                    <div class="info-value gold"><?= $game['rate'] ?>x</div>
                </div>
                <div class="info-item">
                    <div class="info-label">Playing Date</div>
                    <div class="info-value"><?= date('d M Y', strtotime($playingDate)) ?></div>
                </div>
                <div class="info-item">
                    <div class="info-label">Your Balance</div>
                    <div class="info-value green">₹<?= number_format($balance, 2) ?></div>
                </div>
            </div>
        </div>

        <!-- Bid Type Tabs -->
        <div class="bid-tabs">
            <div class="bid-tab active" onclick="switchTab('jodi')">
                <div class="tab-icon">🎯</div>
                <div class="tab-name">JODI</div>
                <div class="tab-rate"><?= $game['rate'] ?>x</div>
            </div>
            <div class="bid-tab" onclick="switchTab('andar')">
                <div class="tab-icon">⬅️</div>
                <div class="tab-name">ANDAR</div>
                <div class="tab-rate"><?= $game['rate']/10 ?>x</div>
            </div>
            <div class="bid-tab" onclick="switchTab('bahar')">
                <div class="tab-icon">➡️</div>
                <div class="tab-name">BAHAR</div>
                <div class="tab-rate"><?= $game['rate']/10 ?>x</div>
            </div>
        </div>

        <!-- JODI Section -->
        <div class="bid-section" id="jodiSection">
            <div class="section-title">🎯 Enter Jodi (00-99)</div>
            <div class="jodi-row">
                <input type="text" id="jodiNumber" class="jodi-number" maxlength="2" placeholder="00" inputmode="numeric">
                <input type="number" id="jodiAmount" class="amount-input" placeholder="Enter Amount" min="<?= $min_bid ?>" max="<?= $max_bid ?>">
                <button type="button" onclick="addJodiBid()" class="btn-add">+ ADD</button>
            </div>
            <div class="min-max">Min: ₹<?= $min_bid ?> | Max: ₹<?= $max_bid ?></div>
            <div class="hint-box">
                💡 <strong>Jodi:</strong> Enter 2 digit number (00-99). If result matches, you win <?= $game['rate'] ?>x your bid amount!
            </div>
        </div>

        <!-- ANDAR Section -->
        <div class="bid-section" id="andarSection" style="display: none;">
            <div class="section-title">⬅️ Select Andar Digit (0-9)</div>
            <div class="digit-grid" id="andarGrid">
                <?php for ($i = 0; $i <= 9; $i++): ?>
                    <button type="button" class="digit-btn" onclick="selectAndar(<?= $i ?>)"><?= $i ?></button>
                <?php endfor; ?>
            </div>
            <div class="amount-row">
                <input type="number" id="andarAmount" class="full-input" placeholder="Enter Amount" min="<?= $min_bid ?>" max="<?= $max_bid ?>">
                <button type="button" onclick="addAndarBid()" class="btn-add">+ ADD</button>
            </div>
            <div class="min-max">Min: ₹<?= $min_bid ?> | Max: ₹<?= $max_bid ?></div>
            <div class="hint-box">
                💡 <strong>Andar:</strong> Select first digit. Example: Andar 5 wins on 50,51,52,53,54,55,56,57,58,59. Win <?= $game['rate']/10 ?>x!
            </div>
        </div>

        <!-- BAHAR Section -->
        <div class="bid-section" id="baharSection" style="display: none;">
            <div class="section-title">➡️ Select Bahar Digit (0-9)</div>
            <div class="digit-grid" id="baharGrid">
                <?php for ($i = 0; $i <= 9; $i++): ?>
                    <button type="button" class="digit-btn" onclick="selectBahar(<?= $i ?>)"><?= $i ?></button>
                <?php endfor; ?>
            </div>
            <div class="amount-row">
                <input type="number" id="baharAmount" class="full-input" placeholder="Enter Amount" min="<?= $min_bid ?>" max="<?= $max_bid ?>">
                <button type="button" onclick="addBaharBid()" class="btn-add">+ ADD</button>
            </div>
            <div class="min-max">Min: ₹<?= $min_bid ?> | Max: ₹<?= $max_bid ?></div>
            <div class="hint-box">
                💡 <strong>Bahar:</strong> Select second digit. Example: Bahar 5 wins on 05,15,25,35,45,55,65,75,85,95. Win <?= $game['rate']/10 ?>x!
            </div>
        </div>

        <!-- Bids List -->
        <div class="bids-container">
            <div class="bids-header">
                <div class="bids-title">📝 Your Bids</div>
                <div class="bids-count" id="bidCount">0</div>
            </div>
            <div class="bids-list" id="bidsList">
                <div class="bids-empty">No bids added yet. Add your first bid above!</div>
            </div>
            <div class="bids-total">
                <span class="total-label">Total Amount:</span>
                <span class="total-amount">₹<span id="totalAmount">0</span></span>
            </div>
        </div>

        <!-- Captcha & Submit -->
        <form method="POST" id="bidForm">
            <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
            <input type="hidden" name="place_bid" value="1">
            <input type="hidden" name="bids_data" id="bidsData">
            
            <div class="captcha-box">
                <div class="captcha-question"><?= $num1 ?> + <?= $num2 ?> = ?</div>
                <input type="number" name="captcha_answer" class="captcha-input" placeholder="?" required>
            </div>
            
            <!-- Daily Limit Info -->
            <div style="text-align:center;font-size:12px;color:#888;margin-bottom:10px;">
                Daily Bid Limit: ₹<?= number_format($dailyBidLimit, 0) ?>
            </div>
            
            <button type="submit" class="btn-submit" id="submitBtn" disabled>🎰 PLACE BID</button>
        </form>
        
        <!-- Today's Bids (Cancellable) -->
        <?php if (!empty($todayBids)): ?>
        <div style="margin-top:30px;padding:15px;background:#1a1a1a;border-radius:12px;">
            <div style="font-size:14px;color:#ffd700;margin-bottom:15px;">📋 Today's Bids (<?= count($todayBids) ?>)</div>
            <?php foreach ($todayBids as $bid): 
                $canCancel = canCancelBid($pdo, $bid['id'], $user['id']);
            ?>
            <div style="display:flex;justify-content:space-between;align-items:center;padding:10px;background:#0f0f0f;border-radius:8px;margin-bottom:8px;">
                <div>
                    <span style="color:#ffd700;"><?= ucfirst($bid['bid_type']) ?></span>
                    <span style="color:#fff;font-weight:bold;margin-left:10px;"><?= $bid['number'] ?></span>
                    <span style="color:#00ff88;margin-left:10px;">₹<?= number_format($bid['amount'], 0) ?></span>
                </div>
                <?php if ($canCancel['can_cancel']): ?>
                <form method="POST" style="display:inline;">
                    <input type="hidden" name="csrf_token" value="<?= $csrfToken ?>">
                    <input type="hidden" name="bid_id" value="<?= $bid['id'] ?>">
                    <button type="submit" name="cancel_bid" style="background:#ff4444;color:white;border:none;padding:5px 10px;border-radius:5px;font-size:11px;cursor:pointer;" onclick="return confirm('Cancel this bid? Amount will be refunded.')">Cancel</button>
                </form>
                <?php else: ?>
                <span style="color:#888;font-size:11px;">Cannot cancel</span>
                <?php endif; ?>
            </div>
            <?php endforeach; ?>
            <div style="font-size:11px;color:#888;margin-top:10px;">⏰ Cancel allowed until <?= $bidCancelMinutes ?> minutes before close time</div>
        </div>
        <?php endif; ?>
    </div>

    <!-- Confirmation Modal -->
    <div class="modal" id="confirmModal">
        <div class="modal-content">
            <div class="modal-title">⚠️ Confirm Your Bids</div>
            <div class="modal-info">
                <div class="modal-row">
                    <span class="modal-label">Current Balance</span>
                    <span class="modal-value green">₹<?= number_format($balance, 2) ?></span>
                </div>
                <div class="modal-row">
                    <span class="modal-label">Total Bid</span>
                    <span class="modal-value" id="modalTotal">₹0</span>
                </div>
                <div class="modal-row">
                    <span class="modal-label">After Balance</span>
                    <span class="modal-value gold" id="modalAfter">₹<?= number_format($balance, 2) ?></span>
                </div>
            </div>
            <div class="modal-bids" id="modalBids"></div>
            <div class="modal-actions">
                <button class="modal-btn cancel" onclick="closeModal()">Cancel</button>
                <button class="modal-btn confirm" onclick="confirmSubmit()">Confirm</button>
            </div>
        </div>
    </div>

    <script>
        let bids = [];
        let currentTab = 'jodi';
        let selectedAndar = null;
        let selectedBahar = null;
        const rate = <?= $game['rate'] ?>;
        const minBid = <?= $min_bid ?>;
        const maxBid = <?= $max_bid ?>;
        const walletBalance = <?= $balance ?>;

        // Tab Switching
        function switchTab(tab) {
            currentTab = tab;
            document.querySelectorAll('.bid-tab').forEach((t, i) => {
                t.classList.remove('active');
                if ((tab === 'jodi' && i === 0) || (tab === 'andar' && i === 1) || (tab === 'bahar' && i === 2)) {
                    t.classList.add('active');
                }
            });
            document.getElementById('jodiSection').style.display = tab === 'jodi' ? 'block' : 'none';
            document.getElementById('andarSection').style.display = tab === 'andar' ? 'block' : 'none';
            document.getElementById('baharSection').style.display = tab === 'bahar' ? 'block' : 'none';
        }

        // Select Andar Digit
        function selectAndar(digit) {
            selectedAndar = digit;
            document.querySelectorAll('#andarGrid .digit-btn').forEach((btn, i) => {
                btn.classList.toggle('selected', i === digit);
            });
        }

        // Select Bahar Digit
        function selectBahar(digit) {
            selectedBahar = digit;
            document.querySelectorAll('#baharGrid .digit-btn').forEach((btn, i) => {
                btn.classList.toggle('selected', i === digit);
            });
        }

        // Only allow numbers in jodi input
        document.getElementById('jodiNumber').addEventListener('input', function() {
            this.value = this.value.replace(/[^0-9]/g, '');
        });

        // Add Jodi Bid
        function addJodiBid() {
            let number = document.getElementById('jodiNumber').value;
            let amount = parseFloat(document.getElementById('jodiAmount').value);
            
            if (!number || number.length === 0) {
                alert('Enter Jodi number!');
                return;
            }
            
            number = number.padStart(2, '0');
            if (parseInt(number) > 99) {
                alert('Jodi must be 00-99');
                return;
            }
            
            if (!amount || amount < minBid || amount > maxBid) {
                alert('Amount must be ₹' + minBid + ' to ₹' + maxBid);
                return;
            }
            
            addBid('jodi', number, amount);
            document.getElementById('jodiNumber').value = '';
            document.getElementById('jodiAmount').value = '';
        }

        // Add Andar Bid
        function addAndarBid() {
            if (selectedAndar === null) {
                alert('Select a digit first!');
                return;
            }
            
            let amount = parseFloat(document.getElementById('andarAmount').value);
            if (!amount || amount < minBid || amount > maxBid) {
                alert('Amount must be ₹' + minBid + ' to ₹' + maxBid);
                return;
            }
            
            addBid('andar', selectedAndar.toString(), amount);
            document.getElementById('andarAmount').value = '';
        }

        // Add Bahar Bid
        function addBaharBid() {
            if (selectedBahar === null) {
                alert('Select a digit first!');
                return;
            }
            
            let amount = parseFloat(document.getElementById('baharAmount').value);
            if (!amount || amount < minBid || amount > maxBid) {
                alert('Amount must be ₹' + minBid + ' to ₹' + maxBid);
                return;
            }
            
            addBid('bahar', selectedBahar.toString(), amount);
            document.getElementById('baharAmount').value = '';
        }

        // Add bid to list
        function addBid(type, number, amount) {
            // Check duplicate
            let exists = bids.find(b => b.type === type && b.number === number);
            if (exists) {
                alert('This bid already added!');
                return;
            }
            
            bids.push({ type, number, amount });
            renderBids();
        }

        // Remove bid
        function removeBid(index) {
            bids.splice(index, 1);
            renderBids();
        }

        // Render bids list
        function renderBids() {
            let html = '';
            let total = 0;
            
            if (bids.length === 0) {
                html = '<div class="bids-empty">No bids added yet. Add your first bid above!</div>';
            } else {
                bids.forEach((bid, index) => {
                    let winAmount = bid.type === 'jodi' ? bid.amount * rate : bid.amount * (rate / 10);
                    total += bid.amount;
                    
                    html += `
                        <div class="bid-item">
                            <div class="bid-left">
                                <span class="bid-type ${bid.type}">${bid.type.toUpperCase()}</span>
                                <span class="bid-number">${bid.number}</span>
                            </div>
                            <div class="bid-right">
                                <div class="bid-amount">₹${bid.amount}</div>
                                <div class="bid-win">Win: ₹${winAmount.toFixed(0)}</div>
                            </div>
                            <button onclick="removeBid(${index})" class="bid-remove">✕</button>
                        </div>
                    `;
                });
            }
            
            document.getElementById('bidsList').innerHTML = html;
            document.getElementById('bidCount').textContent = bids.length;
            document.getElementById('totalAmount').textContent = total.toFixed(0);
            document.getElementById('submitBtn').disabled = bids.length === 0;
            document.getElementById('bidsData').value = JSON.stringify(bids);
        }

        // Form submit - show confirmation
        document.getElementById('bidForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            if (bids.length === 0) {
                alert('Add at least one bid!');
                return;
            }
            
            let total = bids.reduce((sum, b) => sum + b.amount, 0);
            if (total > walletBalance) {
                alert('Insufficient balance!');
                return;
            }
            
            // Show modal
            document.getElementById('modalTotal').textContent = '₹' + total.toFixed(0);
            document.getElementById('modalAfter').textContent = '₹' + (walletBalance - total).toFixed(2);
            
            let bidsHtml = '';
            bids.forEach(bid => {
                let win = bid.type === 'jodi' ? bid.amount * rate : bid.amount * (rate / 10);
                bidsHtml += `<div class="modal-bid">
                    <span>${bid.type.toUpperCase()} - ${bid.number}</span>
                    <span>₹${bid.amount} → Win ₹${win.toFixed(0)}</span>
                </div>`;
            });
            document.getElementById('modalBids').innerHTML = bidsHtml;
            
            document.getElementById('confirmModal').classList.add('active');
        });

        function closeModal() {
            document.getElementById('confirmModal').classList.remove('active');
        }

        function confirmSubmit() {
            closeModal();
            document.getElementById('bidForm').submit();
        }
    </script>
</body>
</html>
