<?php
/**
 * Helper Functions - Captcha, Logging, Archive
 * Satta King Platform
 * 
 * NOTE: This file should be included AFTER auth.php in admin pages
 */

// =============================================
// CAPTCHA FUNCTIONS (Fixed - Multiple Form Support)
// =============================================

/**
 * Generate Admin Captcha (simple math) - UNIQUE PER FORM
 * @param string $formId Unique identifier for the form (e.g., 'payment', 'bid', 'delete')
 * @return string HTML for captcha display
 */
function generateAdminCaptcha($formId = 'default') {
    $num1 = rand(1, 20);
    $num2 = rand(1, 20);
    $answer = $num1 + $num2;
    
    // Store in session with unique key per form
    if (!isset($_SESSION['admin_captcha'])) {
        $_SESSION['admin_captcha'] = [];
    }
    $_SESSION['admin_captcha'][$formId] = $answer;
    
    return "<span style='color:#ffd700;font-weight:bold;'>$num1 + $num2 = ?</span>";
}

/**
 * Verify Admin Captcha - CHECKS SPECIFIC FORM OR ANY FORM
 * @param string $answer User's captcha answer
 * @param string|null $formId Form identifier (null = check all forms)
 * @return bool
 */
function verifyAdminCaptcha($answer, $formId = null) {
    if (!isset($_SESSION['admin_captcha']) || empty($_SESSION['admin_captcha'])) {
        return false;
    }
    
    $userAnswer = intval($answer);
    
    // If specific form ID provided, check only that
    if ($formId !== null) {
        if (!isset($_SESSION['admin_captcha'][$formId])) {
            return false;
        }
        $expected = intval($_SESSION['admin_captcha'][$formId]);
        unset($_SESSION['admin_captcha'][$formId]);
        return $userAnswer === $expected;
    }
    
    // If no form ID, check against ALL stored captchas (for backward compatibility)
    foreach ($_SESSION['admin_captcha'] as $key => $expected) {
        if ($userAnswer === intval($expected)) {
            unset($_SESSION['admin_captcha'][$key]);
            return true;
        }
    }
    
    return false;
}

/**
 * Get Captcha Error Message
 */
function getCaptchaError() {
    return '❌ Wrong captcha answer! Please try again.';
}

/**
 * Verify Inline/JS Captcha (lenient mode for inline buttons)
 * This is for buttons that use JavaScript prompt captcha
 * Accepts any answer that looks like a valid sum (2-40)
 * @param string $answer User's captcha answer
 * @return bool
 */
function verifyInlineCaptcha($answer) {
    $userAnswer = intval($answer);
    // Valid sum of two numbers 1-20 would be between 2 and 40
    return $userAnswer >= 2 && $userAnswer <= 40;
}

/**
 * Generate HIDDEN captcha data for JavaScript prompts
 * Returns array with num1, num2, answer for client-side verification
 */
function generateJsCaptcha() {
    $num1 = rand(1, 20);
    $num2 = rand(1, 20);
    return [
        'num1' => $num1,
        'num2' => $num2,
        'answer' => $num1 + $num2
    ];
}

// =============================================
// USER CAPTCHA FUNCTIONS (for Login/Register)
// =============================================

/**
 * Generate User Captcha (simple math for login/register forms)
 * @return string HTML for captcha display
 */
function generateUserCaptcha() {
    $num1 = rand(1, 15);
    $num2 = rand(1, 15);
    $answer = $num1 + $num2;
    $_SESSION['user_captcha_answer'] = $answer;
    return "$num1 + $num2 = ?";
}

/**
 * Verify User Captcha
 * @param string $answer User's captcha answer
 * @return bool
 */
function verifyUserCaptcha($answer) {
    if (!isset($_SESSION['user_captcha_answer'])) {
        return false;
    }
    $expected = intval($_SESSION['user_captcha_answer']);
    // Don't unset so captcha persists on failed login attempts
    return intval($answer) === $expected;
}

/**
 * Clear User Captcha after successful verification
 */
function clearUserCaptcha() {
    unset($_SESSION['user_captcha_answer']);
}

// =============================================
// ACTIVITY LOG FUNCTION
// =============================================

/**
 * Log Activity with Archive Names
 */
function logActivity($pdo, $admin_id, $action_type, $table_name, $record_id, $old_data, $new_data, $description, $user_id = null) {
    try {
        $admin_name = null;
        if ($admin_id) {
            $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
            $stmt->execute([$admin_id]);
            $admin_name = $stmt->fetchColumn();
        }
        
        $user_name = null;
        if ($user_id) {
            $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user_name = $stmt->fetchColumn();
        }
        
        $stmt = $pdo->prepare("INSERT INTO activity_log 
            (user_id, user_name_archive, admin_id, admin_name_archive, action_type, table_name, record_id, old_data, new_data, description, ip_address) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $user_id, $user_name, $admin_id, $admin_name, $action_type, $table_name, $record_id,
            $old_data ? json_encode($old_data) : null,
            $new_data ? json_encode($new_data) : null,
            $description, $_SERVER['REMOTE_ADDR'] ?? null
        ]);
    } catch (Exception $e) {
        // Silent fail
    }
}

// =============================================
// CALCULATION LOG FUNCTION
// =============================================

/**
 * Log Calculation with Archive Names
 */
function logCalculation($pdo, $user_id, $action_type, $ref_type, $ref_id, $amount, $balance_before, $balance_after, $formula, $description, $performed_by = null, $game_name = null) {
    try {
        $user_name = null;
        if ($user_id) {
            $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
            $stmt->execute([$user_id]);
            $user_name = $stmt->fetchColumn();
        }
        
        $performer_name = null;
        if ($performed_by) {
            $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
            $stmt->execute([$performed_by]);
            $performer_name = $stmt->fetchColumn();
        }
        
        $stmt = $pdo->prepare("INSERT INTO calculation_log 
            (user_id, user_name_archive, action_type, reference_type, reference_id, game_name_archive, amount, balance_before, balance_after, calculation_formula, description, performed_by, performed_by_name) 
            VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
        $stmt->execute([
            $user_id, $user_name, $action_type, $ref_type, $ref_id, $game_name,
            $amount, $balance_before, $balance_after, $formula, $description, $performed_by ?? $user_id, $performer_name
        ]);
    } catch (Exception $e) {
        // Silent fail
    }
}

// =============================================
// ARCHIVE HELPER FUNCTIONS
// =============================================

/**
 * Get User Name (with archive fallback)
 */
function getUserName($pdo, $user_id) {
    $stmt = $pdo->prepare("SELECT name FROM users WHERE id = ?");
    $stmt->execute([$user_id]);
    $name = $stmt->fetchColumn();
    
    if ($name) return $name;
    
    $stmt = $pdo->prepare("SELECT name FROM users_archive WHERE id = ?");
    $stmt->execute([$user_id]);
    $name = $stmt->fetchColumn();
    
    return $name ?: "User #$user_id (Deleted)";
}

/**
 * Get Game Name (with archive fallback)
 */
function getGameName($pdo, $game_id) {
    $stmt = $pdo->prepare("SELECT name FROM games WHERE id = ?");
    $stmt->execute([$game_id]);
    $name = $stmt->fetchColumn();
    
    if ($name) return $name;
    
    $stmt = $pdo->prepare("SELECT name FROM games_archive WHERE id = ?");
    $stmt->execute([$game_id]);
    $name = $stmt->fetchColumn();
    
    return $name ?: "Game #$game_id (Deleted)";
}

/**
 * Store archive names before creating records
 */
function storeArchiveNames($pdo, $table, $id, $user_id = null, $game_id = null) {
    $updates = [];
    $params = [];
    
    if ($user_id) {
        $stmt = $pdo->prepare("SELECT name, mobile FROM users WHERE id = ?");
        $stmt->execute([$user_id]);
        $user = $stmt->fetch();
        if ($user) {
            $updates[] = "user_name_archive = ?";
            $updates[] = "user_mobile_archive = ?";
            $params[] = $user['name'];
            $params[] = $user['mobile'];
        }
    }
    
    if ($game_id) {
        $stmt = $pdo->prepare("SELECT name FROM games WHERE id = ?");
        $stmt->execute([$game_id]);
        $game_name = $stmt->fetchColumn();
        if ($game_name) {
            $updates[] = "game_name_archive = ?";
            $params[] = $game_name;
        }
    }
    
    if (!empty($updates)) {
        $params[] = $id;
        $sql = "UPDATE $table SET " . implode(", ", $updates) . " WHERE id = ?";
        $stmt = $pdo->prepare($sql);
        $stmt->execute($params);
    }
}
?>
