<?php
/**
 * Settings API - App configuration, Payment info
 * Endpoints:
 *   GET /api/settings.php?action=app_config
 *   GET /api/settings.php?action=payment_info
 */

require_once 'config.php';

$action = $_GET['action'] ?? 'app_config';

switch ($action) {
    case 'app_config':
        handleAppConfig($pdo);
        break;
    case 'payment_info':
        handlePaymentInfo($pdo);
        break;
    case 'contact':
        handleContactInfo($pdo);
        break;
    case 'rules':
        handleGameRules($pdo);
        break;
    default:
        sendError('Invalid action', 400);
}

/**
 * Get App Configuration
 */
function handleAppConfig($pdo) {
    // Public endpoint - no auth required
    
    $config = [
        'app' => [
            'name' => getSetting($pdo, 'site_name', 'Satta King'),
            'version' => '1.0.0',
            'min_version' => '1.0.0',
            'force_update' => false,
            'maintenance_mode' => getSetting($pdo, 'maintenance_mode', '0') == '1',
            'maintenance_message' => getSetting($pdo, 'maintenance_message', 'App is under maintenance. Please try again later.')
        ],
        'rates' => [
            'jodi' => (int)getSetting($pdo, 'jodi_rate', '95'),
            'andar' => round((int)getSetting($pdo, 'jodi_rate', '95') / 10, 1),
            'bahar' => round((int)getSetting($pdo, 'jodi_rate', '95') / 10, 1)
        ],
        'limits' => [
            'min_bid' => (int)getSetting($pdo, 'min_bid', '10'),
            'max_bid' => (int)getSetting($pdo, 'max_bid', '10000'),
            'daily_bid_limit' => (int)getSetting($pdo, 'daily_bid_limit', '100000'),
            'min_deposit' => (int)getSetting($pdo, 'min_deposit', '100'),
            'max_deposit' => (int)getSetting($pdo, 'max_deposit', '50000'),
            'min_withdraw' => (int)getSetting($pdo, 'min_withdraw', '100'),
            'max_withdraw' => (int)getSetting($pdo, 'max_withdraw', '50000'),
            'daily_withdraw_limit' => (int)getSetting($pdo, 'daily_withdraw_limit', '50000')
        ],
        'features' => [
            'referral_enabled' => true,
            'transaction_pin_required' => getSetting($pdo, 'require_transaction_pin', '1') == '1',
            'bid_cancellation_enabled' => true,
            'bid_cancel_minutes' => (int)getSetting($pdo, 'bid_cancel_minutes', '5')
        ],
        'referral' => [
            'referrer_bonus' => (int)getSetting($pdo, 'referral_bonus_referrer', '50'),
            'referred_bonus' => (int)getSetting($pdo, 'referral_bonus_referred', '25')
        ],
        'currency' => [
            'code' => 'INR',
            'symbol' => '₹'
        ]
    ];
    
    sendResponse($config);
}

/**
 * Get Payment Info for Deposits
 */
function handlePaymentInfo($pdo) {
    // This could be public or require auth based on your needs
    
    $paymentInfo = [
        'upi' => [
            'enabled' => true,
            'upi_id' => getSetting($pdo, 'payment_upi_id', ''),
            'upi_name' => getSetting($pdo, 'payment_upi_name', 'Satta King')
        ],
        'bank' => [
            'enabled' => getSetting($pdo, 'bank_transfer_enabled', '1') == '1',
            'bank_name' => getSetting($pdo, 'payment_bank_name', ''),
            'account_number' => getSetting($pdo, 'payment_account_no', ''),
            'account_holder' => getSetting($pdo, 'payment_account_holder', ''),
            'ifsc_code' => getSetting($pdo, 'payment_ifsc', '')
        ],
        'qr_code' => getSetting($pdo, 'payment_qr_code', '') ? '/uploads/' . getSetting($pdo, 'payment_qr_code', '') : null,
        'instructions' => [
            '1. Make payment using UPI or Bank Transfer',
            '2. Take screenshot of successful payment',
            '3. Submit deposit request with screenshot',
            '4. Wait for admin approval (usually within 15 minutes)'
        ],
        'note' => getSetting($pdo, 'payment_note', 'Please verify payment details before transferring.')
    ];
    
    sendResponse($paymentInfo);
}

/**
 * Get Contact Information
 */
function handleContactInfo($pdo) {
    $contact = [
        'whatsapp' => getSetting($pdo, 'contact_whatsapp', ''),
        'telegram' => getSetting($pdo, 'contact_telegram', ''),
        'email' => getSetting($pdo, 'contact_email', ''),
        'phone' => getSetting($pdo, 'contact_phone', ''),
        'support_hours' => getSetting($pdo, 'support_hours', '10:00 AM - 10:00 PM'),
        'social' => [
            'instagram' => getSetting($pdo, 'social_instagram', ''),
            'facebook' => getSetting($pdo, 'social_facebook', ''),
            'youtube' => getSetting($pdo, 'social_youtube', '')
        ]
    ];
    
    sendResponse($contact);
}

/**
 * Get Game Rules
 */
function handleGameRules($pdo) {
    $rate = (int)getSetting($pdo, 'jodi_rate', '95');
    
    $rules = [
        'game_types' => [
            [
                'type' => 'jodi',
                'name' => 'Jodi',
                'description' => 'Guess the 2-digit number (00-99)',
                'rate' => $rate,
                'example' => "Bid ₹10 on 45, if result is 45, win ₹" . (10 * $rate)
            ],
            [
                'type' => 'andar',
                'name' => 'Andar',
                'description' => 'Guess the first digit of result (0-9)',
                'rate' => round($rate / 10, 1),
                'example' => "Bid ₹10 on 4, if result is 45, win ₹" . (10 * $rate / 10)
            ],
            [
                'type' => 'bahar',
                'name' => 'Bahar',
                'description' => 'Guess the second digit of result (0-9)',
                'rate' => round($rate / 10, 1),
                'example' => "Bid ₹10 on 5, if result is 45, win ₹" . (10 * $rate / 10)
            ]
        ],
        'general_rules' => [
            'Results are published at game close time',
            'Bids cannot be placed after game closes',
            'Winning amount is automatically added to wallet',
            'Minimum withdrawal amount is ₹' . getSetting($pdo, 'min_withdraw', '100'),
            'All decisions by admin are final'
        ],
        'bid_cancellation' => [
            'enabled' => true,
            'time_limit' => (int)getSetting($pdo, 'bid_cancel_minutes', '5'),
            'description' => 'Bids can be cancelled up to ' . getSetting($pdo, 'bid_cancel_minutes', '5') . ' minutes before game closes'
        ]
    ];
    
    sendResponse($rules);
}
?>
