<?php
require_once '../debug.php';
require_once '../config/db.php';
require_once '../config/auth.php';
require_once '../config/functions.php';
require_once '../config/security.php';

initSecureSession();
requireAdmin();

if (!checkSessionTimeout($pdo)) {
    header("Location: ../login.php?msg=Session+expired");
    exit();
}

$admin = getUser($pdo);
trackUserSession($pdo, $admin['id']);

$error = '';
$success = '';

// Activity Log Function

// Handle Actions
if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    
    // Add Game
    if (isset($_POST['add_game'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $name = trim($_POST['name']);
            $open_time = $_POST['open_time'];
            $close_time = $_POST['close_time'];
            $rate = floatval($_POST['rate']);
            $is_overnight = isset($_POST['is_overnight']) ? 1 : 0;
            $status = $_POST['status'];
            $sort_order = intval($_POST['sort_order'] ?? 0);
            
            if (empty($name) || empty($open_time) || empty($close_time) || $rate <= 0) {
                $error = 'All fields are required';
            } else {
                $stmt = $pdo->prepare("INSERT INTO games (name, open_time, close_time, rate, is_overnight, status, sort_order) VALUES (?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([$name, $open_time, $close_time, $rate, $is_overnight, $status, $sort_order]);
                $game_id = $pdo->lastInsertId();
                
                // Log Activity
                logActivity($pdo, $admin['id'], 'CREATE', 'games', $game_id, null, [
                    'name' => $name, 'rate' => $rate, 'status' => $status
                ], "New game '$name' created");
                
                header("Location: games.php?msg=Game+added+successfully");
                exit();
            }
        }
    }
    
    // Edit Game
    if (isset($_POST['edit_game'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $id = $_POST['game_id'];
            
            // Get old data for log
            $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
            $stmt->execute([$id]);
            $old_game = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $name = trim($_POST['name']);
            $open_time = $_POST['open_time'];
            $close_time = $_POST['close_time'];
            $rate = floatval($_POST['rate']);
            $is_overnight = isset($_POST['is_overnight']) ? 1 : 0;
            $status = $_POST['status'];
            $sort_order = intval($_POST['sort_order'] ?? 0);
            
            $stmt = $pdo->prepare("UPDATE games SET name = ?, open_time = ?, close_time = ?, rate = ?, is_overnight = ?, status = ?, sort_order = ? WHERE id = ?");
            $stmt->execute([$name, $open_time, $close_time, $rate, $is_overnight, $status, $sort_order, $id]);
            
            // Log Activity
            logActivity($pdo, $admin['id'], 'UPDATE', 'games', $id, $old_game, [
                'name' => $name, 'rate' => $rate, 'status' => $status
            ], "Game '$name' updated");
            
            header("Location: games.php?msg=Game+updated+successfully");
            exit();
        }
    }
    
    // Delete Game (Archive instead of delete)
    if (isset($_POST['delete_game'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $id = $_POST['game_id'];
        
        // Get game data
        $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
        $stmt->execute([$id]);
        $game = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($game) {
            // Get game statistics
            $stmt = $pdo->prepare("SELECT COUNT(*) as total_bids, COALESCE(SUM(amount), 0) as total_bid_amount FROM bids WHERE game_id = ?");
            $stmt->execute([$id]);
            $bidStats = $stmt->fetch();
            
            $stmt = $pdo->prepare("SELECT COUNT(*) as total_results, COALESCE(SUM(
                (SELECT COALESCE(SUM(win_amount), 0) FROM bids WHERE game_id = ? AND playing_date = r.playing_date AND status = 'won')
            ), 0) as total_payout FROM results r WHERE r.game_id = ?");
            $stmt->execute([$id, $id]);
            $resultStats = $stmt->fetch();
            
            $pdo->beginTransaction();
            try {
                // Archive the game with all stats
                $stmt = $pdo->prepare("INSERT INTO games_archive 
                    (id, name, open_time, close_time, rate, is_overnight, original_status, sort_order, 
                    total_bids, total_bid_amount, total_results, total_payout, 
                    deleted_by, deleted_by_name, original_created_at) 
                    VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)");
                $stmt->execute([
                    $game['id'],
                    $game['name'],
                    $game['open_time'],
                    $game['close_time'],
                    $game['rate'],
                    $game['is_overnight'],
                    $game['status'],
                    $game['sort_order'] ?? 0,
                    $bidStats['total_bids'] ?? 0,
                    $bidStats['total_bid_amount'] ?? 0,
                    $resultStats['total_results'] ?? 0,
                    $resultStats['total_payout'] ?? 0,
                    $admin['id'],
                    $admin['name'],
                    $game['created_at'] ?? date('Y-m-d H:i:s')
                ]);
                
                // Update bids to reference archived game name (denormalize)
                $stmt = $pdo->prepare("UPDATE bids SET game_name_archive = ? WHERE game_id = ? AND game_name_archive IS NULL");
                $stmt->execute([$game['name'], $id]);
                
                // Delete from main table
                $stmt = $pdo->prepare("DELETE FROM games WHERE id = ?");
                $stmt->execute([$id]);
                
                // Log Activity
                logActivity($pdo, $admin['id'], 'DELETE', 'games', $id, $game, null, "Game '{$game['name']}' deleted and archived");
                
                $pdo->commit();
                header("Location: games.php?msg=Game+deleted+and+archived");
                exit();
                
            } catch (Exception $e) {
                $pdo->rollBack();
                $error = 'Error: ' . $e->getMessage();
            }
        }
        }
    }
    
    // Toggle Status
    if (isset($_POST['toggle_status'])) {
        if (!verifyAdminCaptcha($_POST['captcha_answer'] ?? '')) {
            $error = getCaptchaError();
        } else {
            $id = $_POST['game_id'];
            $new_status = $_POST['new_status'];
            
            $stmt = $pdo->prepare("SELECT * FROM games WHERE id = ?");
            $stmt->execute([$id]);
            $old_game = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $stmt = $pdo->prepare("UPDATE games SET status = ? WHERE id = ?");
            $stmt->execute([$new_status, $id]);
            
            // Log Activity
            logActivity($pdo, $admin['id'], 'STATUS_CHANGE', 'games', $id, 
                ['status' => $old_game['status']], 
                ['status' => $new_status], 
                "Game '{$old_game['name']}' status changed to $new_status"
            );
            
            header("Location: games.php?msg=Status+updated");
            exit();
        }
    }
}

// Get Games
$stmt = $pdo->query("SELECT * FROM games ORDER BY sort_order ASC, name ASC");
$games = $stmt->fetchAll();

// Counts
$totalGames = count($games);
$activeGames = count(array_filter($games, fn($g) => $g['status'] == 'active'));
$inactiveGames = $totalGames - $activeGames;
$overnightGames = count(array_filter($games, fn($g) => $g['is_overnight']));

// For sidebar
$stmt = $pdo->query("SELECT COUNT(*) as c FROM users WHERE is_admin = 0 AND status = 'pending'");
$pendingUsers = $stmt->fetch()['c'];
$stmt = $pdo->query("SELECT COUNT(*) as total FROM wallet_requests WHERE status = 'pending'");
$pendingRequests = $stmt->fetch()['total'];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Games - Admin Panel</title>
    <style>
        * { margin: 0; padding: 0; box-sizing: border-box; }
        body { font-family: -apple-system, BlinkMacSystemFont, 'Segoe UI', Roboto, sans-serif; background: #0f0f0f; color: #fff; min-height: 100vh; }
        
        .admin-layout { display: flex; min-height: 100vh; }
        
        .sidebar { width: 260px; background: #1a1a1a; border-right: 1px solid #333; position: fixed; top: 0; left: 0; bottom: 0; overflow-y: auto; z-index: 100; }
        .sidebar-header { padding: 20px; border-bottom: 1px solid #333; text-align: center; }
        .sidebar-logo { font-size: 22px; font-weight: 800; color: #ffd700; }
        .sidebar-subtitle { font-size: 11px; color: #888; margin-top: 5px; }
        .sidebar-nav { padding: 15px 0; }
        .nav-item { display: flex; align-items: center; gap: 12px; padding: 14px 20px; color: #888; text-decoration: none; transition: all 0.3s; border-left: 3px solid transparent; }
        .nav-item:hover { background: rgba(255,255,255,0.05); color: #fff; }
        .nav-item.active { background: rgba(139,0,0,0.3); color: #ffd700; border-left-color: #ffd700; }
        .nav-icon { font-size: 20px; width: 24px; text-align: center; }
        .nav-text { font-size: 14px; font-weight: 500; }
        .nav-badge { margin-left: auto; background: #ff4444; color: #fff; padding: 2px 8px; border-radius: 10px; font-size: 11px; font-weight: 700; }
        
        .main-content { flex: 1; margin-left: 260px; padding: 20px; }
        
        .top-bar { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 20px; border-bottom: 1px solid #333; flex-wrap: wrap; gap: 15px; }
        .page-title { font-size: 24px; font-weight: 700; }
        .page-title span { color: #ffd700; }
        .btn-add { padding: 12px 25px; background: linear-gradient(135deg, #00c853 0%, #00a844 100%); border: none; border-radius: 10px; color: #fff; font-weight: 700; cursor: pointer; font-size: 14px; }
        
        .alert { padding: 15px; border-radius: 10px; margin-bottom: 20px; font-size: 14px; }
        .alert.success { background: rgba(0,255,0,0.2); border: 1px solid #00cc00; color: #00ff00; }
        .alert.error { background: rgba(255,0,0,0.2); border: 1px solid #ff4444; color: #ff6666; }
        
        .stats-row { display: grid; grid-template-columns: repeat(auto-fit, minmax(150px, 1fr)); gap: 15px; margin-bottom: 25px; }
        .stat-box { background: #1a1a1a; border-radius: 12px; padding: 20px; text-align: center; border: 1px solid #333; }
        .stat-number { font-size: 28px; font-weight: 700; color: #ffd700; }
        .stat-number.green { color: #00ff88; }
        .stat-number.red { color: #ff4444; }
        .stat-number.blue { color: #3498db; }
        .stat-text { font-size: 12px; color: #888; margin-top: 5px; }
        
        .games-grid { display: grid; grid-template-columns: repeat(auto-fill, minmax(320px, 1fr)); gap: 20px; }
        
        .game-card { background: linear-gradient(145deg, #1e1e1e 0%, #2a2a2a 100%); border-radius: 16px; overflow: hidden; border: 1px solid #333; transition: all 0.3s; }
        .game-card:hover { transform: translateY(-5px); border-color: #ffd700; box-shadow: 0 10px 30px rgba(255,215,0,0.1); }
        .game-card.inactive { opacity: 0.6; }
        
        .game-header { background: linear-gradient(135deg, #8b0000 0%, #5c0000 100%); padding: 18px 20px; display: flex; justify-content: space-between; align-items: center; }
        .game-name { font-size: 18px; font-weight: 700; }
        .game-badges { display: flex; gap: 8px; }
        .badge { padding: 4px 10px; border-radius: 15px; font-size: 10px; font-weight: 700; text-transform: uppercase; }
        .badge.overnight { background: rgba(255,215,0,0.2); color: #ffd700; }
        .badge.active { background: rgba(0,255,136,0.2); color: #00ff88; }
        .badge.inactive { background: rgba(255,68,68,0.2); color: #ff4444; }
        
        .game-body { padding: 20px; }
        .game-stats { display: grid; grid-template-columns: repeat(3, 1fr); gap: 10px; margin-bottom: 20px; }
        .game-stat { background: #1a1a1a; padding: 12px; border-radius: 10px; text-align: center; }
        .game-stat-label { font-size: 10px; color: #666; text-transform: uppercase; }
        .game-stat-value { font-size: 15px; font-weight: 700; color: #fff; margin-top: 4px; }
        .game-stat-value.gold { color: #ffd700; }
        
        .game-actions { display: flex; gap: 10px; padding-top: 15px; border-top: 1px solid #333; }
        .btn-game { flex: 1; padding: 10px; border: none; border-radius: 8px; font-size: 12px; font-weight: 600; cursor: pointer; transition: all 0.3s; }
        .btn-game.edit { background: #333; color: #ffd700; }
        .btn-game.toggle { background: #333; color: #3498db; }
        .btn-game.delete { background: #333; color: #ff4444; }
        .btn-game:hover { opacity: 0.8; }
        
        .modal { display: none; position: fixed; top: 0; left: 0; right: 0; bottom: 0; background: rgba(0,0,0,0.9); z-index: 1000; justify-content: center; align-items: center; padding: 20px; }
        .modal.active { display: flex; }
        .modal-content { background: #1a1a1a; border-radius: 20px; padding: 25px; max-width: 500px; width: 100%; max-height: 90vh; overflow-y: auto; border: 1px solid #333; }
        .modal-header { display: flex; justify-content: space-between; align-items: center; margin-bottom: 25px; padding-bottom: 15px; border-bottom: 1px solid #333; }
        .modal-title { font-size: 20px; font-weight: 700; color: #ffd700; }
        .modal-close { background: none; border: none; color: #888; font-size: 28px; cursor: pointer; line-height: 1; }
        
        .form-group { margin-bottom: 20px; }
        .form-label { display: block; font-size: 13px; color: #888; margin-bottom: 8px; }
        .form-input { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 15px; outline: none; }
        .form-input:focus { border-color: #ffd700; }
        .form-row { display: grid; grid-template-columns: 1fr 1fr; gap: 15px; }
        
        .form-check { display: flex; align-items: center; gap: 10px; padding: 15px; background: #252525; border-radius: 10px; cursor: pointer; }
        .form-check input { width: 20px; height: 20px; accent-color: #ffd700; }
        .form-check-label { font-size: 14px; }
        
        .form-select { width: 100%; padding: 14px; background: #252525; border: 2px solid #333; border-radius: 10px; color: #fff; font-size: 15px; outline: none; }
        .form-select:focus { border-color: #ffd700; }
        .form-select option { background: #252525; }
        
        .btn-submit { width: 100%; padding: 16px; background: linear-gradient(135deg, #00c853 0%, #00a844 100%); border: none; border-radius: 10px; color: #fff; font-size: 16px; font-weight: 700; cursor: pointer; margin-top: 10px; }
        .btn-submit:hover { opacity: 0.9; }
        
        .delete-warning { background: rgba(255,68,68,0.1); border: 1px solid #ff4444; border-radius: 10px; padding: 15px; margin-bottom: 20px; }
        .delete-warning-title { color: #ff4444; font-weight: 700; margin-bottom: 5px; }
        .delete-warning-text { font-size: 13px; color: #888; }
        
        .empty-state { text-align: center; padding: 60px 20px; color: #555; }
        .empty-state .icon { font-size: 60px; margin-bottom: 15px; }
        
        .mobile-nav { display: none; position: fixed; bottom: 0; left: 0; right: 0; background: #1a1a1a; padding: 10px 0; border-top: 1px solid #333; z-index: 100; }
        .mobile-nav-inner { display: flex; justify-content: space-around; }
        .mobile-nav-item { text-align: center; color: #666; text-decoration: none; padding: 5px 10px; }
        .mobile-nav-item.active { color: #ffd700; }
        .mobile-nav-icon { font-size: 20px; display: block; margin-bottom: 3px; }
        .mobile-nav-text { font-size: 10px; }
        
        @media (max-width: 900px) {
            .sidebar { display: none; }
            .main-content { margin-left: 0; padding-bottom: 80px; }
            .mobile-nav { display: block; }
        }
        
        @media (max-width: 500px) {
            .stats-row { grid-template-columns: repeat(2, 1fr); }
            .games-grid { grid-template-columns: 1fr; }
            .form-row { grid-template-columns: 1fr; }
        }
    </style>
</head>
<body>
    <div class="admin-layout">
        <aside class="sidebar">
            <div class="sidebar-header">
                <div class="sidebar-logo">🎰 SATTA KING</div>
                <div class="sidebar-subtitle">Admin Panel</div>
            </div>
            <nav class="sidebar-nav">
                <a href="index.php" class="nav-item"><span class="nav-icon">📊</span><span class="nav-text">Dashboard</span></a>
                <a href="users.php" class="nav-item"><span class="nav-icon">👥</span><span class="nav-text">Users</span><?php if ($pendingUsers > 0): ?><span class="nav-badge"><?= $pendingUsers ?></span><?php endif; ?></a>
                <a href="games.php" class="nav-item active"><span class="nav-icon">🎮</span><span class="nav-text">Games</span></a>
                <a href="bids.php" class="nav-item"><span class="nav-icon">🎯</span><span class="nav-text">All Bids</span></a>
                <a href="results.php" class="nav-item"><span class="nav-icon">🏆</span><span class="nav-text">Results</span></a>
                <a href="wallet.php" class="nav-item"><span class="nav-icon">💰</span><span class="nav-text">Wallet</span><?php if ($pendingRequests > 0): ?><span class="nav-badge"><?= $pendingRequests ?></span><?php endif; ?></a>
                <a href="settings.php" class="nav-item"><span class="nav-icon">⚙️</span><span class="nav-text">Settings</span></a>
                <a href="../logout.php" class="nav-item"><span class="nav-icon">🚪</span><span class="nav-text">Logout</span></a>
            </nav>
        </aside>

        <main class="main-content">
            <div class="top-bar">
                <h1 class="page-title">🎮 <span>Games Management</span></h1>
                <button class="btn-add" onclick="openAddModal()">+ Add New Game</button>
            </div>

            <?php if (isset($_GET['msg'])): ?>
                <div class="alert success"><?= htmlspecialchars($_GET['msg']) ?></div>
            <?php endif; ?>
            <?php if ($error): ?>
                <div class="alert error"><?= $error ?></div>
            <?php endif; ?>

            <div class="stats-row">
                <div class="stat-box">
                    <div class="stat-number"><?= $totalGames ?></div>
                    <div class="stat-text">Total Games</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number green"><?= $activeGames ?></div>
                    <div class="stat-text">Active</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number red"><?= $inactiveGames ?></div>
                    <div class="stat-text">Inactive</div>
                </div>
                <div class="stat-box">
                    <div class="stat-number blue"><?= $overnightGames ?></div>
                    <div class="stat-text">Overnight</div>
                </div>
            </div>

            <?php if (empty($games)): ?>
                <div class="empty-state">
                    <div class="icon">🎮</div>
                    <p>No games found. Add your first game!</p>
                </div>
            <?php else: ?>
                <div class="games-grid">
                    <?php foreach ($games as $game): ?>
                        <div class="game-card <?= $game['status'] == 'inactive' ? 'inactive' : '' ?>">
                            <div class="game-header">
                                <div class="game-name"><?= htmlspecialchars($game['name']) ?></div>
                                <div class="game-badges">
                                    <?php if ($game['is_overnight']): ?>
                                        <span class="badge overnight">🌙 Night</span>
                                    <?php endif; ?>
                                    <span class="badge <?= $game['status'] ?>"><?= $game['status'] ?></span>
                                </div>
                            </div>
                            <div class="game-body">
                                <div class="game-stats">
                                    <div class="game-stat">
                                        <div class="game-stat-label">Open</div>
                                        <div class="game-stat-value"><?= date('h:i A', strtotime($game['open_time'])) ?></div>
                                    </div>
                                    <div class="game-stat">
                                        <div class="game-stat-label">Close</div>
                                        <div class="game-stat-value"><?= date('h:i A', strtotime($game['close_time'])) ?></div>
                                    </div>
                                    <div class="game-stat">
                                        <div class="game-stat-label">Rate</div>
                                        <div class="game-stat-value gold"><?= $game['rate'] ?>x</div>
                                    </div>
                                </div>
                                <div class="game-actions">
                                    <button class="btn-game edit" onclick='openEditModal(<?= json_encode($game) ?>)'>✏️ Edit</button>
                                    <form method="POST" style="flex:1;">
                                        <input type="hidden" name="game_id" value="<?= $game['id'] ?>">
                                        <input type="hidden" name="new_status" value="<?= $game['status'] == 'active' ? 'inactive' : 'active' ?>">
                                        <button type="submit" name="toggle_status" class="btn-game toggle" style="width:100%;">
                                            <?= $game['status'] == 'active' ? '⏸ Disable' : '▶ Enable' ?>
                                        </button>
                                    </form>
                                    <button class="btn-game delete" onclick="openDeleteModal(<?= $game['id'] ?>, '<?= htmlspecialchars($game['name']) ?>')">🗑 Delete</button>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </main>
    </div>

    <!-- Add Game Modal -->
    <div class="modal" id="addModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">➕ Add New Game</div>
                <button class="modal-close" onclick="closeModal('addModal')">&times;</button>
            </div>
            <form method="POST">
                <div class="form-group">
                    <label class="form-label">Game Name</label>
                    <input type="text" name="name" class="form-input" placeholder="e.g. Gali, Desawar" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Open Time</label>
                        <input type="time" name="open_time" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Close Time</label>
                        <input type="time" name="close_time" class="form-input" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Rate (e.g. 95 for 95x)</label>
                        <input type="number" name="rate" class="form-input" placeholder="95" value="95" step="0.1" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Sort Order</label>
                        <input type="number" name="sort_order" class="form-input" placeholder="0" value="0">
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Status</label>
                    <select name="status" class="form-select">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="is_overnight">
                        <span class="form-check-label">🌙 Overnight Game (Closes after midnight)</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('add_game') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                </div>
                <button type="submit" name="add_game" class="btn-submit">Add Game</button>
            </form>
        </div>
    </div>

    <!-- Edit Game Modal -->
    <div class="modal" id="editModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">✏️ Edit Game</div>
                <button class="modal-close" onclick="closeModal('editModal')">&times;</button>
            </div>
            <form method="POST">
                <input type="hidden" name="game_id" id="edit_game_id">
                <div class="form-group">
                    <label class="form-label">Game Name</label>
                    <input type="text" name="name" id="edit_name" class="form-input" required>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Open Time</label>
                        <input type="time" name="open_time" id="edit_open_time" class="form-input" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Close Time</label>
                        <input type="time" name="close_time" id="edit_close_time" class="form-input" required>
                    </div>
                </div>
                <div class="form-row">
                    <div class="form-group">
                        <label class="form-label">Rate</label>
                        <input type="number" name="rate" id="edit_rate" class="form-input" step="0.1" required>
                    </div>
                    <div class="form-group">
                        <label class="form-label">Sort Order</label>
                        <input type="number" name="sort_order" id="edit_sort_order" class="form-input">
                    </div>
                </div>
                <div class="form-group">
                    <label class="form-label">Status</label>
                    <select name="status" id="edit_status" class="form-select">
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <div class="form-group">
                    <label class="form-check">
                        <input type="checkbox" name="is_overnight" id="edit_is_overnight">
                        <span class="form-check-label">🌙 Overnight Game</span>
                    </label>
                </div>
                <div class="form-group">
                    <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('edit_game') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required>
                </div>
                <button type="submit" name="edit_game" class="btn-submit">Update Game</button>
            </form>
        </div>
    </div>

    <!-- Delete Confirmation Modal -->
    <div class="modal" id="deleteModal">
        <div class="modal-content">
            <div class="modal-header">
                <div class="modal-title">🗑️ Delete Game</div>
                <button class="modal-close" onclick="closeModal('deleteModal')">&times;</button>
            </div>
            <div class="delete-warning">
                <div class="delete-warning-title">⚠️ Important</div>
                <div class="delete-warning-text">
                    Game will be archived (not permanently deleted). All transactions and bid history will remain intact.
                </div>
            </div>
            <p style="margin-bottom:20px;">Are you sure you want to delete <strong id="delete_game_name"></strong>?</p>
            <form method="POST">
                <input type="hidden" name="game_id" id="delete_game_id">
                <div class="form-group" style="margin-bottom:15px;">
                    <label class="form-label">🔢 Captcha: <?= generateAdminCaptcha('delete_game') ?></label>
                    <input type="number" name="captcha_answer" class="form-input" placeholder="Enter answer" required style="background:#1a1a1a;border:1px solid #333;color:#fff;padding:10px;border-radius:8px;width:100%;">
                </div>
                <div style="display:flex;gap:15px;">
                    <button type="button" class="btn-submit" style="background:#333;" onclick="closeModal('deleteModal')">Cancel</button>
                    <button type="submit" name="delete_game" class="btn-submit" style="background:linear-gradient(135deg,#ff4444,#cc0000);">Delete & Archive</button>
                </div>
            </form>
        </div>
    </div>

    <!-- Mobile Nav -->
    <nav class="mobile-nav">
        <div class="mobile-nav-inner">
            <a href="index.php" class="mobile-nav-item"><span class="mobile-nav-icon">📊</span><span class="mobile-nav-text">Home</span></a>
            <a href="users.php" class="mobile-nav-item"><span class="mobile-nav-icon">👥</span><span class="mobile-nav-text">Users</span></a>
            <a href="games.php" class="mobile-nav-item active"><span class="mobile-nav-icon">🎮</span><span class="mobile-nav-text">Games</span></a>
            <a href="wallet.php" class="mobile-nav-item"><span class="mobile-nav-icon">💰</span><span class="mobile-nav-text">Wallet</span></a>
            <a href="settings.php" class="mobile-nav-item"><span class="mobile-nav-icon">⚙️</span><span class="mobile-nav-text">Settings</span></a>
        </div>
    </nav>

    <script>
        function openAddModal() {
            document.getElementById('addModal').classList.add('active');
        }
        
        function openEditModal(game) {
            document.getElementById('edit_game_id').value = game.id;
            document.getElementById('edit_name').value = game.name;
            document.getElementById('edit_open_time').value = game.open_time;
            document.getElementById('edit_close_time').value = game.close_time;
            document.getElementById('edit_rate').value = game.rate;
            document.getElementById('edit_sort_order').value = game.sort_order || 0;
            document.getElementById('edit_status').value = game.status;
            document.getElementById('edit_is_overnight').checked = game.is_overnight == 1;
            document.getElementById('editModal').classList.add('active');
        }
        
        function openDeleteModal(gameId, gameName) {
            document.getElementById('delete_game_id').value = gameId;
            document.getElementById('delete_game_name').textContent = gameName;
            document.getElementById('deleteModal').classList.add('active');
        }
        
        function closeModal(id) {
            document.getElementById(id).classList.remove('active');
        }
        
        document.querySelectorAll('.modal').forEach(modal => {
            modal.addEventListener('click', function(e) {
                if (e.target === this) this.classList.remove('active');
            });
        });
    </script>
</body>
</html>